package fetcher

import (
	"bytes"
	"context"
	"encoding/json"
	"fmt"
	"math/rand"
	"os"
	"os/exec"
	"path/filepath"
	"strconv"

	"a.yandex-team.ru/security/xray/internal/procutils"
)

type skynetFile struct {
	Name string
	Type string
	Size int64
}

func (f *Fetcher) skynetTorrentInfo(ctx context.Context, rbtorrent string) ([]skynetFile, error) {
	command := exec.CommandContext(ctx, "sky", "files", "--timeout=60", "--json", rbtorrent)
	result, err := procutils.RunCmd(command)
	if err != nil {
		return nil, err
	}

	if result.ExitCode != 0 {
		return nil, fmt.Errorf("unexpected exit status %d: %s", result.ExitCode, string(bytes.TrimSpace(result.Stderr)))
	}

	var files []skynetFile
	if err = json.Unmarshal(result.Stdout, &files); err != nil {
		return nil, fmt.Errorf("unmarshal: %w", err)
	}

	return files, nil
}

func (f *Fetcher) skynetTorrentDownload(ctx context.Context, rbtorrent, dst string) error {
	// TODO(buglloc): do it in the separated container!
	command := exec.CommandContext(
		ctx,
		"sky", "get",
		"--deduplicate=No",
		"--timeout", strconv.FormatFloat(FetchTimeout.Seconds(), 'f', -1, 64),
		"--dir", dst,
		"-w", rbtorrent,
	)
	result, err := procutils.RunCmd(command)
	if err != nil {
		return err
	}

	if result.ExitCode != 0 {
		return fmt.Errorf("unexpected exit status %d: %s", result.ExitCode, string(result.Stderr))
	}
	return nil
}

func (f *Fetcher) skynetDownload(ctx context.Context, uri URI, dst string) (int64, error) {
	if f.singleFileMode {
		return f.skynetDownloadFile(ctx, uri, dst)
	}

	return f.skynetDownloadDir(ctx, uri, dst)
}

func (f *Fetcher) skynetDownloadDir(ctx context.Context, uri URI, dst string) (int64, error) {
	// TODO(buglloc): do it in the separated container!
	files, err := f.skynetTorrentInfo(ctx, uri.Target)
	if err != nil {
		return 0, fmt.Errorf("get rbtorrent info: %w", err)
	}

	var size int64
	for _, f := range files {
		size += f.Size
	}
	if size > f.maxSize {
		return 0, fmt.Errorf("max size exceed %d > %d", files[0].Size, f.maxSize)
	}

	downloadDir := fmt.Sprintf("%s.tmp-%d", dst, rand.Int())
	if err := f.skynetTorrentDownload(ctx, uri.Target, downloadDir); err != nil {
		_ = os.RemoveAll(downloadDir)
		return 0, fmt.Errorf("download rbtorrent: %w", err)
	}

	if err := os.Rename(downloadDir, dst); err != nil {
		return 0, fmt.Errorf("failed to rename tmp-resource from %q to %q: %w", downloadDir, dst, err)
	}

	return size, nil
}

func (f *Fetcher) skynetDownloadFile(ctx context.Context, uri URI, dst string) (int64, error) {
	files, err := f.skynetTorrentInfo(ctx, uri.Target)
	if err != nil {
		return 0, fmt.Errorf("get rbtorrent info: %w", err)
	}

	if len(files) != 1 {
		return 0, fmt.Errorf("expected one file, but got %d", len(files))
	}

	file := files[0]
	if file.Size > f.maxSize {
		return 0, fmt.Errorf("max size exceed %d > %d", files[0].Size, f.maxSize)
	}

	downloadDir := fmt.Sprintf("%s.tmp-%d", dst, rand.Int())
	defer func() { _ = os.RemoveAll(downloadDir) }()

	if err := f.skynetTorrentDownload(ctx, uri.Target, downloadDir); err != nil {
		return 0, fmt.Errorf("download rbtorrent: %w", err)
	}

	tmpLayerPath := filepath.Join(downloadDir, file.Name)
	if err = os.Rename(tmpLayerPath, dst); err != nil {
		return 0, fmt.Errorf("failed to rename tmp-resource from %q to %q: %w", tmpLayerPath, dst, err)
	}

	return file.Size, nil
}
