package rescache

import (
	"container/list"
	"errors"
	"fmt"
	"os"
	"sync"
	"sync/atomic"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/nop"
)

const (
	DefaultMaxSize = 30 * 1024 * 1024 * 1024
	spaceDrift     = 5 * 1024 * 1024 * 1024
)

type Cache struct {
	list    *list.List
	items   map[string]*list.Element
	lock    sync.Mutex
	maxSize int64
	used    int64
	log     log.Logger
}

func NewCache(opts ...Option) *Cache {
	out := &Cache{
		maxSize: DefaultMaxSize,
		list:    list.New(),
		items:   make(map[string]*list.Element),
		log:     &nop.Logger{},
	}

	for _, opt := range opts {
		opt(out)
	}
	return out
}

func (c *Cache) Fetch(uri string, fetch func() (*Resource, error)) (*Resource, error) {
	c.lock.Lock()

	item, ok := c.items[uri]
	if ok && !resourceExists(item.Value.(*Resource).Path) {
		l := item.Value.(*Resource)
		c.log.Error("cached path doesn't exists", log.String("id", l.ID), log.String("path", l.Path))
		c.dropItem(item)
		ok = false
	}

	if !ok {
		c.lock.Unlock()

		resource, err := fetch()
		if err != nil {
			return nil, err
		}

		out, err := c.Store(resource)
		if err != nil {
			c.log.Error("can't save resource to cache", log.String("id", resource.ID), log.Error(err))
		}

		out.track()
		return out, nil
	}

	resource := item.Value.(*Resource)
	resource.track()

	c.log.Info("got resource from cache", log.String("id", resource.ID))
	c.list.MoveToBack(item)
	c.lock.Unlock()

	return resource, nil
}

func (c *Cache) Store(resource *Resource) (*Resource, error) {
	c.log.Info("append resource to cache", log.String("id", resource.ID))

	c.lock.Lock()
	defer c.lock.Unlock()

	if i, ok := c.items[resource.ID]; ok {
		return i.Value.(*Resource), nil
	}

	if c.maxSize < resource.Bytes {
		return resource, fmt.Errorf("resource %s is too big: %d > %d (max)", resource.ID, resource.Bytes, c.maxSize)
	}

	if c.maxSize < resource.Bytes+c.used {
		elem := c.list.Front()
		for c.maxSize-spaceDrift < resource.Bytes+c.used {
			if elem == nil {
				if c.maxSize < resource.Bytes+c.used {
					return resource, errors.New("resource storage fulfilled, but couldn't find any stale resources")
				}
				break
			}

			rLayer := elem.Value.(*Resource)
			if atomic.LoadInt32(&rLayer.refCount) > 0 {
				elem = elem.Next()
				continue
			}

			c.log.Info("storage cache fulfilled, remove resource",
				log.String("id", rLayer.ID),
				log.Int64("max_size", c.maxSize), log.Int64("res_size", resource.Bytes), log.Int64("used", c.used))

			if err := os.RemoveAll(rLayer.Path); err != nil {
				c.log.Error("can't resource resource from cache",
					log.String("id", rLayer.ID),
					log.Error(err))
			}

			c.dropItem(elem)
			elem = c.list.Front()
		}
	}

	item := c.list.PushBack(resource)
	c.used += resource.Bytes
	c.items[resource.ID] = item
	return resource, nil
}

func (c *Cache) dropItem(e *list.Element) {
	c.list.Remove(e)
	l := e.Value.(*Resource)
	delete(c.items, l.ID)
	c.used -= l.Bytes
}

func resourceExists(resPath string) bool {
	if _, err := os.Stat(resPath); err == nil {
		return true
	}
	return false
}
