package resstorage

import (
	"context"
	"fmt"
	"io/fs"
	"io/ioutil"
	"os"
	"path"
	"path/filepath"
	"sort"
	"strings"

	"golang.org/x/sync/singleflight"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/nop"
	"a.yandex-team.ru/security/xray/internal/fsutil"
	"a.yandex-team.ru/security/xray/internal/storage/fetcher"
	"a.yandex-team.ru/security/xray/internal/storage/rescache"
)

const DefaultMaxSize = 30 * 1024 * 1024 * 1024

type Storage struct {
	storagePath string
	fetcher     *fetcher.Fetcher
	cache       *rescache.Cache
	fetchGroup  singleflight.Group
	log         log.Logger
}

func NewStorage(storagePath string, opts ...Option) (*Storage, error) {
	if err := fsutil.CreateDir(storagePath); err != nil {
		return nil, fmt.Errorf("failed to create storage dir: %w", err)
	}

	storage := &Storage{
		storagePath: storagePath,
		log:         &nop.Logger{},
	}

	cacheOpts := []rescache.Option{
		rescache.WithMaxSize(DefaultMaxSize),
	}
	var fetcherOpts []fetcher.Option

	for _, opt := range opts {
		switch o := opt.(type) {
		case maxSizeOption:
			cacheOpts = append(cacheOpts, rescache.WithMaxSize(o.maxSize))
		case loggerOption:
			storage.log = o.log
			cacheOpts = append(cacheOpts, rescache.WithLogger(o.log))
			fetcherOpts = append(fetcherOpts, fetcher.WithLogger(o.log))
		}
	}

	storage.fetcher = fetcher.NewFetcher(fetcherOpts...)
	storage.cache = rescache.NewCache(cacheOpts...)
	return storage, storage.Promote()
}

func (s *Storage) Download(ctx context.Context, uri fetcher.URI) (*rescache.Resource, error) {
	return s.cache.Fetch(uri.ID, func() (*rescache.Resource, error) {
		layerPath := s.makePath(uri.ID)
		size, err := s.fetcher.Download(ctx, uri, layerPath)
		if err != nil {
			return nil, err
		}

		return &rescache.Resource{
			ID:    uri.ID,
			Bytes: size,
			Path:  layerPath,
		}, nil
	})
}

func (s *Storage) Promote() error {
	calcDirSize := func(target string) (int64, error) {
		var size int64
		err := filepath.WalkDir(target, func(osPathname string, d fs.DirEntry, err error) error {
			if err != nil {
				return err
			}

			if d.IsDir() {
				return nil
			}

			st, err := os.Stat(osPathname)
			size += st.Size()

			return err
		})
		return size, err
	}

	resources, err := ioutil.ReadDir(s.storagePath)
	if err != nil {
		return fmt.Errorf("failed to promote storage: %w", err)
	}

	sort.Slice(resources, func(i, j int) bool {
		return resources[i].ModTime().UnixNano() < resources[j].ModTime().UnixNano()
	})

	for _, l := range resources {
		fileName := l.Name()
		if !strings.HasPrefix(fileName, "res-") {
			continue
		}

		resPath := path.Join(s.storagePath, fileName)
		if strings.Contains(fileName, ".tmp-") {
			s.log.Info("remove temporary resource", log.String("name", fileName))
			_ = os.RemoveAll(resPath)
			continue
		}

		size, err := calcDirSize(resPath)
		if err != nil {
			s.log.Error("can't calculate resource size, remove it", log.Error(err))
			_ = os.RemoveAll(resPath)
			continue
		}

		_, _ = s.cache.Store(&rescache.Resource{
			ID:    fileName[4:],
			Path:  resPath,
			Bytes: size,
		})
	}
	return nil
}

func (s *Storage) Close() {
	s.fetcher.Close()
}

func (s *Storage) makePath(id string) string {
	return filepath.Join(s.storagePath, fmt.Sprintf("res-%s", id))
}

func (s *Storage) ParseURI(ctx context.Context, uri string) (fetcher.URI, error) {
	// TODO(anton-k): DRY
	parsedURI, err := s.fetcher.ParseURI(ctx, uri)
	if err != nil {
		return fetcher.URI{}, err
	}

	return parsedURI, nil
}
