package s3storage

import (
	"context"
	"crypto/hmac"
	"crypto/sha256"
	"encoding/hex"
	"fmt"
	"io"
	"net/http"
	"net/http/httputil"
	"strings"
	"time"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/s3"
	"github.com/aws/aws-sdk-go/service/s3/s3manager"
	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/yandex/awstvm"
	"a.yandex-team.ru/library/go/yandex/tvm"
	"a.yandex-team.ru/security/libs/go/yahttp"
	"a.yandex-team.ru/security/xray/pkg/checks/check"
	"a.yandex-team.ru/security/xray/pkg/collectors/collector"
)

const (
	region      = "yandex"
	contentType = "text/plain; charset=UTF-8"
)

type Storage struct {
	cfg           *Options
	s3Client      *s3.S3
	uploader      *s3manager.Uploader
	downloader    *s3manager.Downloader
	downloadProxy *httputil.ReverseProxy
}

type Options struct {
	Endpoint    string
	Bucket      string
	AccessKeyID string
	TvmID       tvm.ClientID
	SignKey     string
}

func NewS3Storage(tvmClient tvm.Client, opts *Options) (*Storage, error) {
	storage := &Storage{
		cfg: opts,
	}

	if err := storage.initS3Client(tvmClient); err != nil {
		return nil, fmt.Errorf("failed to initialize S3 client: %w", err)
	}

	if err := storage.initBucket(); err != nil {
		return nil, fmt.Errorf("failed to initialize S3 bucket: %w", err)
	}

	storage.initUploader()

	storage.initDownloader()

	storage.initDownloadProxy()

	return storage, nil
}

func (s *Storage) initDownloadProxy() {
	host := fmt.Sprintf("%s.%s", s.cfg.Bucket, s.cfg.Endpoint)

	director := func(req *http.Request) {
		req.URL.Scheme = "https"
		req.URL.Host = host
	}

	s.downloadProxy = &httputil.ReverseProxy{
		Director:  director,
		Transport: s.s3Client.Config.HTTPClient.Transport,
	}
}

func (s *Storage) initBucket() error {
	if _, err := s.s3Client.HeadBucket(&s3.HeadBucketInput{Bucket: aws.String(s.cfg.Bucket)}); err != nil {
		_, err := s.s3Client.CreateBucket(&s3.CreateBucketInput{Bucket: aws.String(s.cfg.Bucket)})
		if err != nil {
			return err
		}
	}
	return nil
}

func (s *Storage) initS3Client(tvmClient tvm.Client) error {
	creds, err := awstvm.NewS3Credentials(tvmClient,
		awstvm.WithTvmClientID(s.cfg.TvmID),
		awstvm.WithAccessKeyID(s.cfg.AccessKeyID),
	)
	if err != nil {
		return fmt.Errorf("create credentials: %w", err)
	}

	cfg := aws.NewConfig().
		WithRegion(region).
		WithEndpoint(s.cfg.Endpoint).
		WithCredentials(creds).
		WithHTTPClient(yahttp.NewClient(yahttp.Config{
			RedirectPolicy: yahttp.RedirectNoFollow,
			Timeout:        time.Minute * 2,
			DialTimeout:    time.Second,
		}))

	s3Session, err := session.NewSession()
	if err != nil {
		err = fmt.Errorf("failed to create session: %w", err)
		return err
	}

	s.s3Client = s3.New(s3Session, cfg)
	return nil
}

func (s *Storage) initUploader() {
	s.uploader = s3manager.NewUploaderWithClient(s.s3Client)
}

func (s *Storage) initDownloader() {
	s.downloader = s3manager.NewDownloaderWithClient(s.s3Client)
}

func (s *Storage) UploadFile(ctx context.Context, path string, data io.Reader) error {
	_, err := s.uploader.UploadWithContext(
		ctx,
		&s3manager.UploadInput{
			Bucket:      aws.String(s.cfg.Bucket),
			Key:         aws.String(path),
			Body:        data,
			ContentType: aws.String(contentType),
		},
	)
	return err
}

func (s *Storage) UploadLog(ctx context.Context, analyzeID string, data io.Reader) (logPath string, err error) {
	logPath = s.makeLogPath(analyzeID)
	err = s.UploadFile(ctx, logPath, data)
	return
}

func (s *Storage) UploadResult(ctx context.Context, analyzeID string, data io.Reader) (resultsPath string, err error) {
	resultsPath = s.makeResultPath(analyzeID)
	err = s.UploadFile(ctx, resultsPath, data)
	return
}

func (s *Storage) UploadCheckResults(ctx context.Context, check check.Check, id string, data io.Reader) (err error) {
	checkResultsPath := s.makeCheckResultsPath(check, id)
	err = s.UploadFile(ctx, checkResultsPath, data)
	return
}

func (s *Storage) UploadCollectorResults(ctx context.Context, collector collector.Collector, id string, data io.Reader) (err error) {
	collectorResultsPath := s.makeCollectorResultsPath(collector, id)
	err = s.UploadFile(ctx, collectorResultsPath, data)
	return
}

func (s *Storage) ProxyFile(path string, response *echo.Response) error {
	req, err := http.NewRequest("GET", s.fileURI(path, false), nil)
	if err != nil {
		return fmt.Errorf("failed to create proxy request: %w", err)
	}

	s.downloadProxy.ServeHTTP(response, req)
	return nil
}

func (s *Storage) FileURI(path string) string {
	return s.fileURI(path, true)
}

func (s *Storage) fileURI(path string, secure bool) string {
	scheme := "http"
	if secure {
		scheme = "https"
	}
	return fmt.Sprintf("%s://%s.%s/%s", scheme, s.cfg.Bucket, s.cfg.Endpoint, path)
}

func (s *Storage) DownloadFile(path string) ([]byte, error) {
	result := &aws.WriteAtBuffer{}
	_, err := s.downloader.Download(
		result,
		&s3.GetObjectInput{
			Bucket: aws.String(s.cfg.Bucket),
			Key:    aws.String(path),
		},
	)

	return result.Bytes(), err
}

func (s *Storage) DownloadCheckResults(check check.Check, id string) ([]byte, error) {
	path := s.makeCheckResultsPath(check, id)
	return s.DownloadFile(path)
}

func (s *Storage) DownloadCollectorResults(collector collector.Collector, id string) ([]byte, error) {
	path := s.makeCollectorResultsPath(collector, id)
	return s.DownloadFile(path)
}

func (s *Storage) DeleteFolder(folder string) error {
	prefix := strings.Trim(folder, "/") + "/"
	iter := s3manager.NewDeleteListIterator(s.s3Client, &s3.ListObjectsInput{
		Bucket: aws.String(s.cfg.Bucket),
		Prefix: aws.String(prefix),
	})

	if err := s3manager.NewBatchDeleteWithClient(s.s3Client).Delete(aws.BackgroundContext(), iter); err != nil {
		return fmt.Errorf("failed to delete objects: %w", err)
	}

	return nil
}

func (s *Storage) makeLogPath(analyzeID string) string {
	h := hmac.New(sha256.New, []byte(s.cfg.SignKey))
	_, _ = h.Write([]byte(analyzeID))
	filename := hex.EncodeToString(h.Sum(nil))
	return fmt.Sprintf("%s.log.txt", filename)
}

func (s *Storage) makeResultPath(analyzeID string) string {
	h := hmac.New(sha256.New, []byte(s.cfg.SignKey))
	_, _ = h.Write([]byte(analyzeID))
	filename := hex.EncodeToString(h.Sum(nil))
	return fmt.Sprintf("%s.result.pb", filename)
}

func (s *Storage) makeCheckResultsPath(check check.Check, id string) string {
	return fmt.Sprintf("%s/%s.result.pb", check.Type(), id)
}

func (s *Storage) makeCollectorResultsPath(collector collector.Collector, id string) string {
	return fmt.Sprintf("collectors/%s/%s/%s.result.pb", collector.Type(), collector.Version(), id)
}
