package boxer

import (
	"bytes"
	"context"
	"encoding/base64"
	"fmt"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/xray/internal/boxer"
	"a.yandex-team.ru/security/xray/internal/boxer/boxer_out"
	"a.yandex-team.ru/security/xray/pkg/checks/check"
	boxer_col "a.yandex-team.ru/security/xray/pkg/collectors/boxer"
	"a.yandex-team.ru/security/xray/pkg/xrayrpc"
)

const (
	Name = "Boxer"
	Type = "boxer"
)

var (
	_          check.BoxFSCheck = (*Boxer)(nil)
	_          check.Check      = (*Boxer)(nil)
	collectors                  = []string{boxer_col.Type}
)

type (
	Boxer struct {
		command   string
		collector string
	}
)

func New(_ check.Config) *Boxer {
	return &Boxer{}
}

func (s *Boxer) Name() string {
	return Name
}

func (s *Boxer) Type() string {
	return Type
}

func (s *Boxer) Collectors() []string {
	return collectors
}

func (s *Boxer) Sync(_ context.Context) error {
	return nil
}

func (s *Boxer) Deadline() time.Duration {
	return 1 * time.Minute
}

func (s *Boxer) ProcessCollectorResults(_ log.Logger, result map[string]*xrayrpc.CollectorResult) (check.Issues, error) {
	boxerInfo := result[boxer_col.Type].Finding.GetBoxer()
	if boxerInfo == nil {
		return nil, nil
	}

	boxerResult, err := boxer.ProcessInfo(&boxer_out.CollectorResult{Finding: boxerInfo})
	if err != nil {
		return nil, fmt.Errorf("failed to process boxer info: %w", err)
	}
	if boxerResult.Issue == nil {
		return nil, nil
	}

	return []*xrayrpc.Issue{{
		Id:       generateIssueID(boxerResult.Issue),
		Severity: kindToSeverity(boxerResult.Issue.Kind),
		Kind:     xrayrpc.IssueKind_IK_SECURITY,
		Details: &xrayrpc.Issue_Boxer{
			Boxer: boxerResult.Issue,
		},
	}}, nil
}

func kindToSeverity(kind xrayrpc.BoxerIssueDetail_Kind) xrayrpc.Severity {
	switch kind {
	case xrayrpc.BoxerIssueDetail_BIK_OBSOLETE, xrayrpc.BoxerIssueDetail_BIK_CONFUSED:
		return xrayrpc.Severity_S_HIGH
	default:
		return xrayrpc.Severity_S_UNKNOWN
	}
}

func generateIssueID(issue *xrayrpc.BoxerIssueDetail) string {
	var buf bytes.Buffer
	buf.WriteString(Type)
	buf.WriteByte(':')
	buf.WriteString(issue.Kind.String())

	return base64.RawURLEncoding.EncodeToString(buf.Bytes())
}
