package boxer

import (
	"fmt"

	"a.yandex-team.ru/security/xray/pkg/checks/check"
	"a.yandex-team.ru/security/xray/pkg/xrayrpc"
)

const (
	aboutURL            = "https://wiki.yandex-team.ru/security/x-ray/checks/#boxer"
	obsoleteSummary     = `Box uses an outdated distributive of the %s family (%s)`
	obsoleteDescription = `Using an outdated GNU/Linux distributive may affect the stability and/or security of your application.`
	confusedSummary     = `Box uses an unpopular distributive of the %s family`
	confusedDescription = `The most common GNU/Linux distributive in Yandex is Ubuntu. It should be used to avoid unplanned troubles ;)`
)

func FormatIssue(issue *xrayrpc.Issue) (*check.FormattedIssue, error) {
	details, ok := issue.Details.(*xrayrpc.Issue_Boxer)
	if !ok {
		return nil, check.ErrWrongIssueType
	}

	var out *check.FormattedIssue
	switch details.Boxer.Kind {
	case xrayrpc.BoxerIssueDetail_BIK_OBSOLETE:
		out = &check.FormattedIssue{
			ID:          issue.Id,
			Summary:     fmt.Sprintf(obsoleteSummary, details.Boxer.Distro.Family, details.Boxer.Distro.Release),
			HelpURL:     aboutURL,
			Reference:   details.Boxer.Reference,
			Description: obsoleteDescription,
		}
	case xrayrpc.BoxerIssueDetail_BIK_CONFUSED:
		out = &check.FormattedIssue{
			ID:          issue.Id,
			Summary:     fmt.Sprintf(confusedSummary, details.Boxer.Distro.Family),
			HelpURL:     aboutURL,
			Reference:   details.Boxer.Reference,
			Description: confusedDescription,
		}
	default:
		return nil, check.ErrUnknownKind
	}

	return out, nil
}
