package check

import (
	"context"
	"errors"
	"syscall"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/xray/pkg/xrayrpc"
	"a.yandex-team.ru/yp/go/yson/podagent"
	"a.yandex-team.ru/yp/go/yson/ypapi"
)

type (
	TargetKind int32
)

const (
	TargetKindUnknown TargetKind = iota
	TargetKindStage
	TargetKindDeployUnit
	TargetKindBox
	TargetKindBoxFS
	TargetKindWorkload
)

var (
	ErrWrongIssueType = errors.New("wrong issue type")
	ErrUnknownKind    = errors.New("unknown issue kind")
)

type (
	FormattedIssue struct {
		ID          string
		Summary     string
		HelpURL     string
		Reference   string
		Description string
	}

	Formatter func(*xrayrpc.Issue) (*FormattedIssue, error)

	Issues = []*xrayrpc.Issue

	Check interface {
		// Human readable check name
		Name() string

		// Check type. Needed for filter, parse results and so on.
		Type() string

		// Maximum check time
		Deadline() time.Duration

		// Periodic sync db data
		Sync(ctx context.Context) error
	}

	StageCheck interface {
		Check

		CheckStage(logger log.Logger, stage *StageSpec) (Issues, error)
	}

	DeployUnitCheck interface {
		Check

		CheckDeployUnit(logger log.Logger, deployUnit *DeployUnitSpec) (Issues, error)
	}

	BoxCheck interface {
		Check

		CheckBox(logger log.Logger, box *BoxSpec) (Issues, error)
	}

	BoxFSCheck interface {
		Check

		// ProcessCollectorResults results
		ProcessCollectorResults(logger log.Logger, result map[string]*xrayrpc.CollectorResult) (Issues, error)

		// Must return collectors needed
		Collectors() []string
	}

	WorkloadCheck interface {
		Check

		CheckWorkload(logger log.Logger, workload *WorkloadSpec) (Issues, error)
	}

	Config struct {
		AuthToken    string
		ContainerDir string
		HostDir      string
		Logger       log.Logger
	}

	ContainerRequirements struct {
		// Require network
		Net bool

		// Require bind mounts
		// E.g.: /home/tmp /tmp
		Bind []string

		// Environments variables
		Env []string
	}

	ContainerResult struct {
		Status  syscall.WaitStatus
		Stdout  []byte
		Stderr  []byte
		Elapsed time.Duration
	}

	StageSpec struct {
		Stage *ypapi.TStage
		Path  *xrayrpc.TargetPath
	}

	DeployUnitSpec struct {
		Stage *StageSpec
		Name  string
		Spec  *ypapi.TDeployUnitSpec
		Path  *xrayrpc.TargetPath
	}

	BoxSpec struct {
		DeployUnit *DeployUnitSpec
		Pod        *podagent.TPodAgentSpec
		Spec       *podagent.TBox
		Path       *xrayrpc.TargetPath
	}

	WorkloadSpec struct {
		Box  *BoxSpec
		Spec *podagent.TWorkload
		Path *xrayrpc.TargetPath
	}
)

func (c *ContainerRequirements) PortoNet() string {
	if c.Net {
		// TODO(buglloc): ooops!
		return "host"
	}
	return "none"
}
