package checks

import (
	"fmt"

	"a.yandex-team.ru/security/xray/pkg/checks/boxer"
	"a.yandex-team.ru/security/xray/pkg/checks/check"
	"a.yandex-team.ru/security/xray/pkg/checks/flakyimage"
	"a.yandex-team.ru/security/xray/pkg/checks/flakysource"
	"a.yandex-team.ru/security/xray/pkg/checks/gracefulshutdown"
	"a.yandex-team.ru/security/xray/pkg/checks/log4jfinder"
	"a.yandex-team.ru/security/xray/pkg/checks/probelover"
	"a.yandex-team.ru/security/xray/pkg/checks/safenodes"
	"a.yandex-team.ru/security/xray/pkg/checks/secretsearch"
	"a.yandex-team.ru/security/xray/pkg/checks/secretstorage"
	"a.yandex-team.ru/security/xray/pkg/checks/soxer"
	"a.yandex-team.ru/security/xray/pkg/checks/spring4shellfinder"
	"a.yandex-team.ru/security/xray/pkg/checks/yadios"
	"a.yandex-team.ru/security/xray/pkg/checks/yodax"
	"a.yandex-team.ru/security/xray/pkg/xrayrpc"
)

type Checks map[check.TargetKind][]check.Check

func NewCheck(typ string, env check.Config) (Checks, error) {
	switch typ {
	case secretsearch.Type:
		ssCheck := secretsearch.New(env)
		return Checks{
			check.TargetKindStage: {
				ssCheck,
			},
			check.TargetKindBoxFS: {
				ssCheck,
			},
		}, nil
	case secretstorage.Type:
		return Checks{
			check.TargetKindDeployUnit: {
				secretstorage.New(env),
			},
		}, nil
	case flakyimage.Type:
		return Checks{
			check.TargetKindBox: {
				flakyimage.New(env),
			},
		}, nil
	case flakysource.Type:
		return Checks{
			check.TargetKindBox: {
				flakysource.New(env),
			},
		}, nil
	case boxer.Type:
		return Checks{
			check.TargetKindBoxFS: {
				boxer.New(env),
			},
		}, nil
	case yodax.Type:
		return Checks{
			check.TargetKindBoxFS: {
				yodax.New(env),
			},
		}, nil
	case yadios.Type:
		return Checks{
			check.TargetKindBoxFS: {
				yadios.New(env),
			},
		}, nil
	case probelover.Type:
		return Checks{
			check.TargetKindWorkload: {
				probelover.New(env),
			},
		}, nil
	case log4jfinder.Type:
		return Checks{
			check.TargetKindBoxFS: {
				log4jfinder.New(env),
			},
		}, nil
	case spring4shellfinder.Type:
		return Checks{
			check.TargetKindBoxFS: {
				spring4shellfinder.New(env),
			},
		}, nil
	default:
		return nil, fmt.Errorf("unsupported check: %s", typ)
	}
}

func NewChecks(env check.Config) Checks {
	ssCheck := secretsearch.New(env)
	return Checks{
		check.TargetKindStage: {
			ssCheck,
			//soxer.New(env),
		},
		check.TargetKindDeployUnit: {
			secretstorage.New(env),
			// TODO(buglloc): enable after Deploy supports SOX stages
			//safenodes.New(env),
		},
		check.TargetKindBox: {
			flakyimage.New(env),
			flakysource.New(env),
		},
		check.TargetKindBoxFS: {
			boxer.New(env),
			ssCheck,
			yodax.New(env),
			yadios.New(env),
			log4jfinder.New(env),
			spring4shellfinder.New(env),
		},
		check.TargetKindWorkload: {
			probelover.New(env),
			// gracefulshutdown.New(env),
		},
	}
}

func ExtractDetails(details interface{}) (string, interface{}) {
	switch v := details.(type) {
	case *xrayrpc.Issue_SecretSearch:
		return secretsearch.Type, v.SecretSearch
	case *xrayrpc.Issue_YadiOs:
		return yadios.Type, v.YadiOs
	case *xrayrpc.Issue_Yodax:
		return yodax.Type, v.Yodax
	case *xrayrpc.Issue_Boxer:
		return boxer.Type, v.Boxer
	case *xrayrpc.Issue_SafeNodes:
		return safenodes.Type, v.SafeNodes
	case *xrayrpc.Issue_FlakySource:
		return flakysource.Type, v.FlakySource
	case *xrayrpc.Issue_FlakyImage:
		return flakyimage.Type, v.FlakyImage
	case *xrayrpc.Issue_ProbeLover:
		return probelover.Type, v.ProbeLover
	case *xrayrpc.Issue_GracefulShutdown:
		return gracefulshutdown.Type, v.GracefulShutdown
	case *xrayrpc.Issue_Soxer:
		return soxer.Type, v.Soxer
	case *xrayrpc.Issue_SecretStorage:
		return secretstorage.Type, v.SecretStorage
	case *xrayrpc.Issue_Log4JFinder:
		return log4jfinder.Type, v.Log4JFinder
	case *xrayrpc.Issue_Spring4ShellFinder:
		return spring4shellfinder.Type, v.Spring4ShellFinder
	default:
		panic(fmt.Sprintf("unexpected issue detail type: %T", v))
	}
}

func FormatIssue(issue *xrayrpc.Issue) (issueType string, formattedIssue *check.FormattedIssue, err error) {
	switch issue.Details.(type) {
	case *xrayrpc.Issue_SecretSearch:
		issueType = secretsearch.Type
		formattedIssue, err = secretsearch.FormatIssue(issue)
	case *xrayrpc.Issue_YadiOs:
		issueType = yadios.Type
		formattedIssue, err = yadios.FormatIssue(issue)
	case *xrayrpc.Issue_Yodax:
		issueType = yodax.Type
		formattedIssue, err = yodax.FormatIssue(issue)
	case *xrayrpc.Issue_Boxer:
		issueType = boxer.Type
		formattedIssue, err = boxer.FormatIssue(issue)
	case *xrayrpc.Issue_SafeNodes:
		issueType = safenodes.Type
		formattedIssue, err = safenodes.FormatIssue(issue)
	case *xrayrpc.Issue_FlakySource:
		issueType = flakysource.Type
		formattedIssue, err = flakysource.FormatIssue(issue)
	case *xrayrpc.Issue_FlakyImage:
		issueType = flakyimage.Type
		formattedIssue, err = flakyimage.FormatIssue(issue)
	case *xrayrpc.Issue_ProbeLover:
		issueType = probelover.Type
		formattedIssue, err = probelover.FormatIssue(issue)
	case *xrayrpc.Issue_GracefulShutdown:
		issueType = gracefulshutdown.Type
		formattedIssue, err = gracefulshutdown.FormatIssue(issue)
	case *xrayrpc.Issue_Soxer:
		issueType = soxer.Type
		formattedIssue, err = soxer.FormatIssue(issue)
	case *xrayrpc.Issue_SecretStorage:
		issueType = secretstorage.Type
		formattedIssue, err = secretstorage.FormatIssue(issue)
	case *xrayrpc.Issue_Log4JFinder:
		issueType = log4jfinder.Type
		formattedIssue, err = log4jfinder.FormatIssue(issue)
	case *xrayrpc.Issue_Spring4ShellFinder:
		issueType = spring4shellfinder.Type
		formattedIssue, err = spring4shellfinder.FormatIssue(issue)
	default:
		err = fmt.Errorf("unexpected issue detail type: %T", issue.Details)
	}

	return
}

func CheckName(checkType string) string {
	switch checkType {
	case secretsearch.Type:
		return secretsearch.Name
	case yadios.Type:
		return yadios.Name
	case yodax.Type:
		return yodax.Name
	case boxer.Type:
		return boxer.Name
	case safenodes.Type:
		return safenodes.Name
	case flakysource.Type:
		return flakysource.Name
	case flakyimage.Type:
		return flakyimage.Name
	case probelover.Type:
		return probelover.Name
	case gracefulshutdown.Type:
		return gracefulshutdown.Name
	case soxer.Type:
		return soxer.Name
	case secretstorage.Type:
		return secretstorage.Name
	case log4jfinder.Type:
		return log4jfinder.Name
	case spring4shellfinder.Type:
		return spring4shellfinder.Name
	default:
		panic(fmt.Sprintf("unknown check type: %s", checkType))
	}
}
