package flakyimage

import (
	"bytes"
	"context"
	"encoding/base64"
	"fmt"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/xray/pkg/checks/check"
	"a.yandex-team.ru/security/xray/pkg/xrayrpc"
)

const (
	Name = "Flaky Image"
	Type = "flaky-image"
)

var (
	_ check.BoxCheck = (*FlakyImage)(nil)
	_ check.Check    = (*FlakyImage)(nil)
)

type (
	FlakyImage struct {
	}
)

func New(_ check.Config) *FlakyImage {
	return &FlakyImage{}
}

func (s *FlakyImage) Name() string {
	return Name
}

func (s *FlakyImage) Type() string {
	return Type
}

func (s *FlakyImage) Sync(_ context.Context) error {
	return nil
}

func (s *FlakyImage) Deadline() time.Duration {
	return 1 * time.Minute
}

func (s *FlakyImage) CheckBox(_ log.Logger, box *check.BoxSpec) (check.Issues, error) {
	if box.DeployUnit.Spec.ImagesForBoxes == nil {
		// no docker images at all, that's fine
		return nil, nil
	}

	image, ok := box.DeployUnit.Spec.ImagesForBoxes[box.Spec.GetId()]
	if !ok {
		// current box doesn't use docker, that's fine
		return nil, nil
	}

	if image.GetTag() == "latest" {
		imageURI := fmt.Sprintf("%s/%s:%s", image.GetRegistryHost(), image.GetName(), image.GetTag())
		return []*xrayrpc.Issue{{
			Kind:     xrayrpc.IssueKind_IK_AVAILABILITY,
			Id:       generateIssueID(box.Spec.GetId()),
			Severity: xrayrpc.Severity_S_MEDIUM,
			Details: &xrayrpc.Issue_FlakyImage{
				FlakyImage: &xrayrpc.FlakyImageIssueDetail{
					Url:         imageURI,
					ProblemKind: xrayrpc.FlakyImageIssueDetail_FIIPK_LATEST_TAG,
				}},
		}}, nil
	}

	return nil, nil
}

func generateIssueID(boxID string) string {
	var buf bytes.Buffer
	buf.WriteString(Type)
	buf.WriteByte(':')
	buf.WriteString(boxID)

	return base64.RawURLEncoding.EncodeToString(buf.Bytes())
}
