package flakysource

import (
	"bytes"
	"context"
	"encoding/base64"
	"fmt"
	"net/url"
	"strings"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/xray/internal/sandbox"
	"a.yandex-team.ru/security/xray/pkg/checks/check"
	"a.yandex-team.ru/security/xray/pkg/xrayrpc"
	"a.yandex-team.ru/yp/go/yson/podagent"
)

const (
	Name = "Flaky Source"
	Type = "flaky-source"
)

var (
	_ check.BoxCheck = (*FlakySource)(nil)
	_ check.Check    = (*FlakySource)(nil)
)

type (
	FlakySource struct {
	}

	Resource interface {
		GetUrl() string
		GetId() string
	}
)

func New(_ check.Config) *FlakySource {
	return &FlakySource{}
}

func (s *FlakySource) Name() string {
	return Name
}

func (s *FlakySource) Type() string {
	return Type
}

func (s *FlakySource) Sync(_ context.Context) error {
	return nil
}

func (s *FlakySource) Deadline() time.Duration {
	return 1 * time.Minute
}

func (s *FlakySource) CheckBox(l log.Logger, box *check.BoxSpec) (check.Issues, error) {
	var issues []*xrayrpc.Issue

	{
		layersIssues, err := checkPortoLayers(l, box)
		switch {
		case err != nil:
			l.Error("porto layers check failed", log.Error(err))
		case layersIssues != nil:
			issues = append(issues, layersIssues...)
		}
	}

	{
		staticResourcesIssues, err := checkStaticResources(l, box)
		switch {
		case err != nil:
			l.Error("static resource check failed", log.Error(err))
		case staticResourcesIssues != nil:
			issues = append(issues, staticResourcesIssues...)
		}
	}

	return issues, nil
}

func checkPortoLayers(l log.Logger, box *check.BoxSpec) ([]*xrayrpc.Issue, error) {
	if box.Spec.Rootfs == nil || len(box.Spec.Rootfs.LayerRefs) == 0 {
		// no porto layers, that's fine
		return nil, nil
	}

	var issues []*xrayrpc.Issue
	for _, layerRef := range box.Spec.Rootfs.LayerRefs {
		var layer *podagent.TLayer
		for _, candidate := range box.Pod.Resources.Layers {
			if candidate.GetId() == layerRef {
				layer = candidate
				break
			}
		}

		if layer == nil {
			l.Error("layer resource was not found", log.String("layer_ref", layerRef))
			continue
		}

		issue, err := checkResource(xrayrpc.FlakySourceIssueDetail_FSISK_LAYER, layer)
		if err != nil {
			l.Error("layer check failed",
				log.String("layer_id", layer.GetId()),
				log.String("layer_url", layer.GetUrl()),
				log.Error(err))
			continue
		}

		if issue != nil {
			issues = append(issues, issue)
		}
	}

	return issues, nil
}

func checkStaticResources(l log.Logger, box *check.BoxSpec) ([]*xrayrpc.Issue, error) {
	if len(box.Spec.StaticResources) == 0 {
		// no static resources, that's fine
		return nil, nil
	}

	var issues []*xrayrpc.Issue
	for _, resourceRef := range box.Spec.StaticResources {
		targetID := resourceRef.GetResourceRef()
		var resource *podagent.TResource
		for _, candidate := range box.Pod.Resources.StaticResources {
			if candidate.GetId() == targetID {
				resource = candidate
				break
			}
		}

		if resource == nil {
			l.Error("static resource was not found", log.String("resource_ref", targetID))
			continue
		}

		issue, err := checkResource(xrayrpc.FlakySourceIssueDetail_FSISK_STATIC_RESOURCE, resource)
		if err != nil {
			l.Error("static resource check failed",
				log.String("layer_id", resource.GetId()),
				log.String("layer_url", resource.GetUrl()),
				log.Error(err))
			continue
		}

		if issue != nil {
			issues = append(issues, issue)
		}
	}

	return issues, nil
}

func checkResource(sourceKind xrayrpc.FlakySourceIssueDetail_SourceKind, resource Resource) (*xrayrpc.Issue, error) {
	uri := resource.GetUrl()
	scheme := strings.SplitN(uri, ":", 2)[0]
	if scheme != "http" && scheme != "https" {
		// not interesting
		return nil, nil
	}

	parsedURI, err := url.Parse(uri)
	if err != nil {
		return nil, fmt.Errorf("failed to parse resource url: %w", err)
	}

	if parsedURI.Host == sandbox.ProxyHost {
		return &xrayrpc.Issue{
			Kind:     xrayrpc.IssueKind_IK_AVAILABILITY,
			Id:       generateIssueID(sourceKind, uri),
			Severity: xrayrpc.Severity_S_MEDIUM,
			Details: &xrayrpc.Issue_FlakySource{
				FlakySource: &xrayrpc.FlakySourceIssueDetail{
					Id:          resource.GetId(),
					Url:         uri,
					SourceKind:  sourceKind,
					ProblemKind: xrayrpc.FlakySourceIssueDetail_FSIPK_SANDBOX_PROXY,
				}},
		}, nil
	}

	return nil, nil
}

func generateIssueID(sourceKind xrayrpc.FlakySourceIssueDetail_SourceKind, id string) string {
	var buf bytes.Buffer
	buf.WriteString(Type)
	buf.WriteByte(':')
	buf.WriteString(sourceKind.String())
	buf.WriteByte(':')
	buf.WriteString(id)

	return base64.RawURLEncoding.EncodeToString(buf.Bytes())
}
