package flakysource

import (
	"fmt"

	"a.yandex-team.ru/security/xray/pkg/checks/check"
	"a.yandex-team.ru/security/xray/pkg/xrayrpc"
)

const (
	aboutURL                 = "https://wiki.yandex-team.ru/security/x-ray/checks/#flakysource"
	layerSourceName          = "box layers"
	staticResourceSourceName = "static resources"
	referenceSbProxy         = "https://wiki.yandex-team.ru/sandbox/proxy/#what-is-this"
	summarySbProxy           = `Sandbox proxy is used as a source of %s`
	descriptionSbProxy       = "Box uses `%s` resource, the source of which is sandbox proxy (`%s`).\n\n" +
		"Never use a sandbox proxy as a source due to its unstable nature. Always prefer `sbr:<resource_id>` sources syntax ;)"
)

func FormatIssue(issue *xrayrpc.Issue) (*check.FormattedIssue, error) {
	details, ok := issue.Details.(*xrayrpc.Issue_FlakySource)
	if !ok {
		return nil, check.ErrWrongIssueType
	}

	var sourceName string
	switch details.FlakySource.SourceKind {
	case xrayrpc.FlakySourceIssueDetail_FSISK_LAYER:
		sourceName = layerSourceName
	case xrayrpc.FlakySourceIssueDetail_FSISK_STATIC_RESOURCE:
		sourceName = staticResourceSourceName
	default:
		return nil, check.ErrUnknownKind
	}

	var out *check.FormattedIssue
	switch details.FlakySource.ProblemKind {
	case xrayrpc.FlakySourceIssueDetail_FSIPK_SANDBOX_PROXY:
		out = &check.FormattedIssue{
			ID:          issue.Id,
			Summary:     fmt.Sprintf(summarySbProxy, sourceName),
			HelpURL:     aboutURL,
			Reference:   referenceSbProxy,
			Description: fmt.Sprintf(descriptionSbProxy, details.FlakySource.Id, details.FlakySource.Url),
		}
	default:
		return nil, check.ErrUnknownKind
	}

	return out, nil
}
