package gracefulshutdown

import (
	"bytes"
	"context"
	"encoding/base64"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/xray/pkg/checks/check"
	"a.yandex-team.ru/security/xray/pkg/xrayrpc"
)

const (
	Name = "Graceful Shutdown"
	Type = "graceful-shutdown"
)

var (
	_ check.WorkloadCheck = (*GracefulShutdown)(nil)
	_ check.Check         = (*GracefulShutdown)(nil)
)

type (
	GracefulShutdown struct {
	}
)

func New(_ check.Config) *GracefulShutdown {
	return &GracefulShutdown{}
}

func (s *GracefulShutdown) Name() string {
	return Name
}

func (s *GracefulShutdown) Type() string {
	return Type
}

func (s *GracefulShutdown) Sync(_ context.Context) error {
	return nil
}

func (s *GracefulShutdown) Deadline() time.Duration {
	return 1 * time.Minute
}

func (s *GracefulShutdown) CheckWorkload(_ log.Logger, workload *check.WorkloadSpec) (check.Issues, error) {
	if workload.Spec.StopPolicy != nil {
		return nil, nil
	}

	return check.Issues{{
		Kind:     xrayrpc.IssueKind_IK_AVAILABILITY,
		Id:       generateIssueID(xrayrpc.GracefulShutdownIssueDetail_GSIPK_NO_STOP_PLOCY),
		Severity: xrayrpc.Severity_S_MEDIUM,
		Details: &xrayrpc.Issue_GracefulShutdown{
			GracefulShutdown: &xrayrpc.GracefulShutdownIssueDetail{
				ProblemKind: xrayrpc.GracefulShutdownIssueDetail_GSIPK_NO_STOP_PLOCY,
			},
		},
	}}, nil
}

func generateIssueID(kind xrayrpc.GracefulShutdownIssueDetail_ProblemKind) string {
	var buf bytes.Buffer
	buf.WriteString(Type)
	buf.WriteByte(':')
	buf.WriteString(kind.String())

	return base64.RawURLEncoding.EncodeToString(buf.Bytes())
}
