package log4jfinder

import (
	"bytes"
	"context"
	"encoding/base64"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/xray/internal/log4j-finder/vuln"
	"a.yandex-team.ru/security/xray/pkg/checks/check"
	"a.yandex-team.ru/security/xray/pkg/collectors/log4jfinder"
	"a.yandex-team.ru/security/xray/pkg/xrayrpc"
)

const (
	Name = "Log4j finder"
	Type = "log4j-finder"
)

var (
	_          check.BoxFSCheck = (*Boxer)(nil)
	_          check.Check      = (*Boxer)(nil)
	collectors                  = []string{log4jfinder.Type}
)

type (
	Boxer struct {
		command string
	}
)

func New(_ check.Config) *Boxer {
	return &Boxer{}
}

func (s *Boxer) Name() string {
	return Name
}

func (s *Boxer) Type() string {
	return Type
}

func (s *Boxer) Command() string {
	return s.command
}

func (s *Boxer) Collectors() []string {
	return collectors
}

func (s *Boxer) Sync(_ context.Context) error {
	return nil
}

func (s *Boxer) Deadline() time.Duration {
	return 5 * time.Minute
}

func (s *Boxer) ProcessCollectorResults(_ log.Logger, result map[string]*xrayrpc.CollectorResult) (check.Issues, error) {
	var badJars []*xrayrpc.Log4JFinderIssueDetail_Log4JInfo
	log4jfinderInfo := result[log4jfinder.Type].Finding.GetLog4JFinder()

	if log4jfinderInfo == nil {
		return nil, nil
	}

	for _, jar := range log4jfinderInfo.Jars {
		if vuln.IsVulnerableVersion(jar.Version) {
			badJars = append(badJars, &xrayrpc.Log4JFinderIssueDetail_Log4JInfo{
				Path:    jar.Path,
				Version: jar.Version,
			})
		}
	}

	if len(badJars) == 0 {
		return nil, nil
	}

	return []*xrayrpc.Issue{{
		Id:       generateIssueID(&xrayrpc.Log4JFinderIssueDetail{}),
		Severity: xrayrpc.Severity_S_HIGH,
		Kind:     xrayrpc.IssueKind_IK_SECURITY,
		Details: &xrayrpc.Issue_Log4JFinder{
			Log4JFinder: &xrayrpc.Log4JFinderIssueDetail{
				Jars: badJars,
			},
		},
	}}, nil
}

func generateIssueID(_ *xrayrpc.Log4JFinderIssueDetail) string {
	var buf bytes.Buffer
	buf.WriteString(Type)

	return base64.RawURLEncoding.EncodeToString(buf.Bytes())
}
