package probelover

import (
	"bytes"
	"context"
	"encoding/base64"
	"time"

	"github.com/google/go-cmp/cmp"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/xray/pkg/checks/check"
	"a.yandex-team.ru/security/xray/pkg/xrayrpc"
	"a.yandex-team.ru/yp/go/yson/podagent"
)

const (
	Name = "Probe Lover"
	Type = "probe-lover"
)

var (
	_ check.WorkloadCheck = (*ProbeLover)(nil)
	_ check.Check         = (*ProbeLover)(nil)
)

type (
	ProbeLover struct {
	}
)

func New(_ check.Config) *ProbeLover {
	return &ProbeLover{}
}

func (s *ProbeLover) Name() string {
	return Name
}

func (s *ProbeLover) Type() string {
	return Type
}

func (s *ProbeLover) Sync(_ context.Context) error {
	return nil
}

func (s *ProbeLover) Deadline() time.Duration {
	return 1 * time.Minute
}

func (s *ProbeLover) CheckWorkload(_ log.Logger, workload *check.WorkloadSpec) (check.Issues, error) {
	if workload.Spec.ReadinessCheck == nil {
		return check.Issues{{
			Kind:     xrayrpc.IssueKind_IK_AVAILABILITY,
			Id:       generateIssueID(xrayrpc.ProbeLoverIssueDetail_PLIPK_NO_READINESS),
			Severity: xrayrpc.Severity_S_MEDIUM,
			Details: &xrayrpc.Issue_ProbeLover{
				ProbeLover: &xrayrpc.ProbeLoverIssueDetail{
					ProblemKind: xrayrpc.ProbeLoverIssueDetail_PLIPK_NO_READINESS,
				},
			},
		}}, nil
	}

	if workload.Spec.LivenessCheck != nil {
		if sameProbes(workload.Spec.ReadinessCheck, workload.Spec.LivenessCheck) {
			return check.Issues{{
				Kind:     xrayrpc.IssueKind_IK_AVAILABILITY,
				Id:       generateIssueID(xrayrpc.ProbeLoverIssueDetail_PLIPK_READINESS_SAME_TO_LIVENESS),
				Severity: xrayrpc.Severity_S_MEDIUM,
				Details: &xrayrpc.Issue_ProbeLover{
					ProbeLover: &xrayrpc.ProbeLoverIssueDetail{
						ProblemKind: xrayrpc.ProbeLoverIssueDetail_PLIPK_READINESS_SAME_TO_LIVENESS,
					},
				},
			}}, nil
		}
	}

	return nil, nil
}

func sameProbes(readiness *podagent.TReadinessCheck, liveness *podagent.TLivenessCheck) bool {
	switch {
	case readiness.Container != nil && liveness.Container == nil:
		return false
	case readiness.Container != nil && liveness.Container != nil:
		return readiness.Container.GetCommandLine() == liveness.Container.GetCommandLine() &&
			readiness.Container.GetCwd() == liveness.Container.GetCwd() &&
			cmp.Equal(readiness.Container.TimeLimit, liveness.Container.TimeLimit)
	case readiness.TcpCheck != nil && liveness.TcpCheck == nil:
		return false
	case readiness.TcpCheck != nil && liveness.TcpCheck != nil:
		return cmp.Equal(readiness.TcpCheck, liveness.TcpCheck)
	case readiness.HttpGet != nil && liveness.HttpGet == nil:
		return false
	case readiness.HttpGet != nil && liveness.HttpGet != nil:
		return readiness.HttpGet.GetPort() == liveness.HttpGet.GetPort() &&
			readiness.HttpGet.GetPath() == liveness.HttpGet.GetPath() &&
			cmp.Equal(readiness.HttpGet.TimeLimit, liveness.HttpGet.TimeLimit)
	default:
		return false
	}
}

func generateIssueID(kind xrayrpc.ProbeLoverIssueDetail_ProblemKind) string {
	var buf bytes.Buffer
	buf.WriteString(Type)
	buf.WriteByte(':')
	buf.WriteString(kind.String())

	return base64.RawURLEncoding.EncodeToString(buf.Bytes())
}
