package safenodes

import (
	"bytes"
	"context"
	"encoding/base64"
	"fmt"
	"regexp"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/xray/internal/abc"
	"a.yandex-team.ru/security/xray/internal/yputil"
	"a.yandex-team.ru/security/xray/pkg/checks/check"
	"a.yandex-team.ru/security/xray/pkg/xrayrpc"
)

const (
	Name = "Safe Nodes"
	Type = "safe-nodes"
)

var (
	_ check.DeployUnitCheck = (*SafeNodes)(nil)
	_ check.Check           = (*SafeNodes)(nil)

	expectedNodeFilter = regexp.MustCompile(`\[/labels/extras/safe_for_sox]\s*=\s*%?true`)
)

type (
	SafeNodes struct {
		abcClient *abc.Client
	}
)

func New(env check.Config) *SafeNodes {
	abcClient, err := abc.NewClient(env.AuthToken)
	if err != nil {
		panic(fmt.Sprintf("failed to create ABC client: %s", err))
	}

	return &SafeNodes{
		abcClient: abcClient,
	}
}

func (s *SafeNodes) Name() string {
	return Name
}

func (s *SafeNodes) Type() string {
	return Type
}

func (s *SafeNodes) Sync(_ context.Context) error {
	return nil
}

func (s *SafeNodes) Deadline() time.Duration {
	return 1 * time.Minute
}

func (s *SafeNodes) CheckDeployUnit(l log.Logger, deployUnit *check.DeployUnitSpec) (check.Issues, error) {
	podSpec := yputil.DeployUnitPodSpec(deployUnit.Spec)
	if podSpec == nil {
		// that's fine
		return nil, nil
	}

	accountID := deployUnit.Stage.Stage.Spec.GetAccountId()
	abcID := yputil.AccountToABC(accountID)
	if abcID == 0 {
		l.Error("failed to parse account ID", log.String("account_id", accountID))
		// TODO(buglloc): return warnings
		return nil, nil
	}

	// TODO(buglloc): too ugly and flaky
	if expectedNodeFilter.FindString(podSpec.GetNodeFilter()) != "" {
		// deploy unit used safe hosts
		return nil, nil
	}

	tags, err := s.abcClient.GetServiceTags(context.Background(), abcID)
	if err != nil {
		l.Error("failed to get ABC account tags", log.Int("abc_id", abcID), log.Error(err))
		// TODO(buglloc): return warnings
		return nil, nil
	}

	if !abc.IsSoxTags(tags) {
		// not a sox service
		return nil, nil
	}

	return []*xrayrpc.Issue{{
		Id:       generateIssueID(xrayrpc.SafeNodesIssueDetail_SNI_EXPECTED_SAFE_NODE),
		Kind:     xrayrpc.IssueKind_IK_SECURITY,
		Severity: xrayrpc.Severity_S_HIGH,
		Details: &xrayrpc.Issue_SafeNodes{
			SafeNodes: &xrayrpc.SafeNodesIssueDetail{
				Kind:       xrayrpc.SafeNodesIssueDetail_SNI_EXPECTED_SAFE_NODE,
				AbcId:      uint32(abcID),
				AbcTags:    abc.CompactTags(tags),
				NodeFilter: podSpec.GetNodeFilter(),
			},
		},
	}}, nil
}

func generateIssueID(kind xrayrpc.SafeNodesIssueDetail_Kind) string {
	var buf bytes.Buffer
	buf.WriteString(Type)
	buf.WriteByte(':')
	buf.WriteString(kind.String())

	return base64.RawURLEncoding.EncodeToString(buf.Bytes())
}
