package secretsearch

import (
	"bytes"
	"context"
	"encoding/base64"
	"encoding/json"
	"fmt"
	"net"
	"os/exec"
	"path/filepath"
	"strings"
	"time"

	"google.golang.org/protobuf/proto"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/ant-secret/secret-search/public/cpp/output/secret_search_out"
	"a.yandex-team.ru/security/xray/internal/ant"
	"a.yandex-team.ru/security/xray/internal/stringutil"
	"a.yandex-team.ru/security/xray/pkg/checks/check"
	"a.yandex-team.ru/security/xray/pkg/collectors/secretsearch"
	"a.yandex-team.ru/security/xray/pkg/xrayrpc"
)

const (
	Name = "Secrets searcher"
	Type = "secret-search"

	antsecretHost = "ant.sec.yandex-team.ru:80"
)

var (
	_ check.StageCheck = (*SecretSearch)(nil)
	_ check.BoxFSCheck = (*SecretSearch)(nil)
	_ check.Check      = (*SecretSearch)(nil)

	collectors = []string{secretsearch.Type}
)

type (
	SecretSearch struct {
		containerCommand string
		stageCommand     stageCommand
		endpointAddr     string
		antClient        *ant.Client
		env              []string
	}

	stageCommand struct {
		Cmd  string
		Args []string
	}
)

var (
	excludes = []string{
		".arc",
		".git",
		".hg",
		".svn",
		".venv",
		".venv3",
		"node_modules",
		"tmp",
		"dev",
		"proc",
		"sys",
		"run",
		"berkanavt",
		"_cacache",
	}
)

func New(env check.Config) *SecretSearch {
	return &SecretSearch{
		containerCommand: strings.Join([]string{
			filepath.Join(env.ContainerDir, "secret-search"),
			"--status-code=3",
			"--format=proto",
			"--num-threads=1",
			"--valid-only",
			"--excludes=" + strings.Join(excludes, ","),
			"/",
		}, " "),
		stageCommand: stageCommand{
			Cmd: filepath.Join(env.HostDir, "secret-search"),
			Args: []string{
				"--status-code=0",
				"--format=proto",
				"--num-threads=1",
				"--valid-only",
				"--check-struct",
				"-",
			},
		},
		antClient: ant.New(env.Logger),
	}
}

func (s *SecretSearch) Name() string {
	return Name
}

func (s *SecretSearch) Type() string {
	return Type
}

func (s *SecretSearch) Collectors() []string {
	return collectors
}

func (s *SecretSearch) Sync(ctx context.Context) error {
	addr, err := net.ResolveTCPAddr("tcp6", antsecretHost)
	if err != nil {
		return fmt.Errorf("failed to resolve ant-secret host %q: %w", antsecretHost, err)
	}

	s.endpointAddr = "http://[" + addr.IP.String() + "]:80/api/v1/validate/"
	s.env = []string{"ANT_VALIDATION_ENDPOINT=" + s.endpointAddr}
	return nil
}

func (s *SecretSearch) Deadline() time.Duration {
	return 10 * time.Minute
}

func (s *SecretSearch) CheckStage(logger log.Logger, stage *check.StageSpec) (check.Issues, error) {
	input, err := json.Marshal(stage)
	if err != nil {
		return nil, fmt.Errorf("failed to marshal stage: %w", err)
	}

	var stderr bytes.Buffer
	cmd := exec.Command(s.stageCommand.Cmd, s.stageCommand.Args...)
	cmd.Stdin = bytes.NewReader(input)
	cmd.Stderr = &stderr
	cmd.Env = s.env
	out, err := cmd.Output()
	if err != nil {
		logger.Error("secret-search exec failed",
			log.String("stderr", stderr.String()),
			log.Error(err),
		)
		return nil, fmt.Errorf("secret-search exec failed: %w", err)
	}

	issues, err := parseOut(out, xrayrpc.SecretSearchIssueDetail_SSIK_STAGE)
	if err != nil {
		logger.Error("unexpected secret-search result", log.String("stdout", string(out)))
		return nil, err
	}

	return issues, nil
}

func (s *SecretSearch) ProcessCollectorResults(l log.Logger, result map[string]*xrayrpc.CollectorResult) (check.Issues, error) {
	var issues check.Issues
	ctx := context.TODO()

	ssDetails := result[secretsearch.Type].Finding.GetSecretSearch()

	if ssDetails == nil {
		return nil, nil
	}

	for _, finding := range ssDetails.SecretsInfo {
		valid, err := s.antClient.IsValidSecret(ctx, finding.Secret, finding.Type)

		if err != nil {
			l.Warn("failed to validate secret", log.String("secret_hash", finding.Hash), log.Error(err))
		}
		if !valid {
			continue
		}

		issues = append(issues, &xrayrpc.Issue{
			Id:       generateIssueID(finding.Hash, parseFindingKind(finding.Kind)),
			Kind:     xrayrpc.IssueKind_IK_SECURITY,
			Severity: xrayrpc.Severity_S_HIGH,
			Details: &xrayrpc.Issue_SecretSearch{
				SecretSearch: &xrayrpc.SecretSearchIssueDetail{
					Kind:      parseFindingKind(finding.Kind),
					Path:      finding.Path,
					Type:      finding.Type,
					Validated: finding.Validated,
					LineNo:    finding.LineNo,
					Secret:    finding.Secret,
				},
			},
		})

	}

	return issues, nil
}

func parseOut(out []byte, kind xrayrpc.SecretSearchIssueDetail_Kind) (check.Issues, error) {
	if len(out) == 0 {
		return nil, nil
	}

	var ssOut secret_search_out.Result
	if err := proto.Unmarshal(out, &ssOut); err != nil {
		return nil, fmt.Errorf("failed to decode secret-search results: %w", err)
	}

	if len(ssOut.Issues) <= 0 {
		// ok
		return nil, nil
	}

	var issues check.Issues
	for _, issue := range ssOut.Issues {
		for _, secret := range issue.Secrets {
			secretHash, ok := secret.Additional["sha1"]
			if !ok {
				secretHash = stringutil.ShaHex(secret.Secret)
			}

			issues = append(issues, &xrayrpc.Issue{
				Id:       generateIssueID(secretHash, kind),
				Kind:     xrayrpc.IssueKind_IK_SECURITY,
				Severity: xrayrpc.Severity_S_HIGH,
				Details: &xrayrpc.Issue_SecretSearch{
					SecretSearch: &xrayrpc.SecretSearchIssueDetail{
						Kind:      kind,
						Path:      issue.Path,
						Type:      secret.Type,
						Validated: secret.Validated,
						LineNo:    secret.LineNo,
						Secret:    secret.Secret,
					},
				},
			})
		}
	}
	return issues, nil
}

func generateIssueID(secret string, kind xrayrpc.SecretSearchIssueDetail_Kind) string {
	var buf bytes.Buffer
	buf.WriteString(Type)
	buf.WriteByte(':')
	buf.WriteString(kind.String())
	buf.WriteByte(';')
	buf.WriteString(secret)

	return base64.RawURLEncoding.EncodeToString(buf.Bytes())
}

func parseFindingKind(kind xrayrpc.SecretSearchFindingDetail_Kind) xrayrpc.SecretSearchIssueDetail_Kind {
	switch kind {
	case xrayrpc.SecretSearchFindingDetail_SSIK_STAGE:
		return xrayrpc.SecretSearchIssueDetail_SSIK_STAGE
	default:
		return xrayrpc.SecretSearchIssueDetail_SSIK_BOX
	}
}
