package secretsearch

import (
	"bytes"
	"fmt"
	"strings"
	"text/template"

	"a.yandex-team.ru/security/xray/pkg/checks/check"
	"a.yandex-team.ru/security/xray/pkg/xrayrpc"
)

const (
	aboutURL   = "https://wiki.yandex-team.ru/security/x-ray/checks/#secretssearcher"
	summary    = `Found "secrets" in %s`
	stageTitle = "stage specification"
	boxTitle   = "box layers"
)

var (
	description, _ = template.New("description").Parse(`
secret info:
  - path: {{.Path}}{{if .LineNo}}#{{.LineNo}}{{end}}
  - secret: ` + "`" + `{{.Secret}}` + "`" + `
  - secret type: {{.Type}}
`)
)

func FormatIssue(issue *xrayrpc.Issue) (*check.FormattedIssue, error) {
	details, ok := issue.Details.(*xrayrpc.Issue_SecretSearch)
	if !ok {
		return nil, check.ErrWrongIssueType
	}

	var issueSummary string
	switch details.SecretSearch.Kind {
	case xrayrpc.SecretSearchIssueDetail_SSIK_STAGE:
		issueSummary = fmt.Sprintf(summary, stageTitle)
	case xrayrpc.SecretSearchIssueDetail_SSIK_BOX:
		issueSummary = fmt.Sprintf(summary, boxTitle)
	default:
		return nil, check.ErrUnknownKind
	}

	var buf bytes.Buffer
	err := description.Execute(&buf, details.SecretSearch)
	if err != nil {
		return nil, fmt.Errorf("failed to format description: %w", err)
	}

	return &check.FormattedIssue{
		ID:          issue.Id,
		Summary:     issueSummary,
		HelpURL:     aboutURL,
		Description: strings.TrimSpace(buf.String()),
	}, nil
}
