package secretstorage

import (
	"bytes"
	"context"
	"encoding/base64"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/xray/internal/yputil"
	"a.yandex-team.ru/security/xray/pkg/checks/check"
	"a.yandex-team.ru/security/xray/pkg/xrayrpc"
)

const (
	Name = "Secrets storage"
	Type = "secrets-storage"
)

var (
	_ check.DeployUnitCheck = (*SecretsStorage)(nil)
	_ check.Check           = (*SecretsStorage)(nil)
)

type SecretsStorage struct{}

func New(_ check.Config) *SecretsStorage {
	return &SecretsStorage{}
}

func (s *SecretsStorage) Name() string {
	return Name
}

func (s *SecretsStorage) Type() string {
	return Type
}

func (s *SecretsStorage) Sync(_ context.Context) error {
	return nil
}

func (s *SecretsStorage) Deadline() time.Duration {
	return 1 * time.Minute
}

func (s *SecretsStorage) CheckDeployUnit(l log.Logger, deployUnit *check.DeployUnitSpec) (check.Issues, error) {
	podSpec := yputil.DeployUnitPodSpec(deployUnit.Spec)
	if podSpec == nil {
		// that's fine
		return nil, nil
	}

	if len(podSpec.Secrets) == 0 {
		// no old secrets
		return nil, nil
	}

	secrets := make([]string, 0, len(podSpec.Secrets))
	for secAlias := range podSpec.Secrets {
		secrets = append(secrets, secAlias)
	}

	return []*xrayrpc.Issue{{
		Id:       generateIssueID(xrayrpc.SecretStorageIssueDetail_SSI_EXPECTED_NEW_STORAGE),
		Kind:     xrayrpc.IssueKind_IK_SECURITY,
		Severity: xrayrpc.Severity_S_HIGH,
		Details: &xrayrpc.Issue_SecretStorage{
			SecretStorage: &xrayrpc.SecretStorageIssueDetail{
				Kind:           xrayrpc.SecretStorageIssueDetail_SSI_EXPECTED_NEW_STORAGE,
				SecretsAliases: secrets,
			},
		},
	}}, nil
}

func generateIssueID(kind xrayrpc.SecretStorageIssueDetail_Kind) string {
	var buf bytes.Buffer
	buf.WriteString(Type)
	buf.WriteByte(':')
	buf.WriteString(kind.String())

	return base64.RawURLEncoding.EncodeToString(buf.Bytes())
}
