package secretstorage

import (
	"fmt"
	"strings"

	"a.yandex-team.ru/security/xray/pkg/checks/check"
	"a.yandex-team.ru/security/xray/pkg/xrayrpc"
)

const (
	aboutURL    = "https://wiki.yandex-team.ru/security/x-ray/checks/#secretsstorage"
	summary     = `Deploy unit uses an old secrets storage`
	description = "Secret%s `%s` is stored in the old secrets storage that is not secure. " +
		"You must use the new storage, please [migrate](https://deploy.yandex-team.ru/docs/how-to/secrets#migraciya-so-staroj-na-novuyu-shemu) to it."
)

func FormatIssue(issue *xrayrpc.Issue) (*check.FormattedIssue, error) {
	details, ok := issue.Details.(*xrayrpc.Issue_SecretStorage)
	if !ok {
		return nil, check.ErrWrongIssueType
	}

	var out *check.FormattedIssue
	switch details.SecretStorage.Kind {
	case xrayrpc.SecretStorageIssueDetail_SSI_UNSPECIFIED, xrayrpc.SecretStorageIssueDetail_SSI_EXPECTED_NEW_STORAGE:
		var multiple string
		if len(details.SecretStorage.SecretsAliases) > 1 {
			multiple = "s"
		}

		out = &check.FormattedIssue{
			ID:          issue.Id,
			Summary:     summary,
			HelpURL:     aboutURL,
			Description: fmt.Sprintf(description, multiple, strings.Join(details.SecretStorage.SecretsAliases, ",")),
		}
	default:
		return nil, check.ErrUnknownKind
	}

	return out, nil
}
