package soxer

import (
	"context"
	"encoding/base64"
	"fmt"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/xray/internal/abc"
	"a.yandex-team.ru/security/xray/internal/yputil"
	"a.yandex-team.ru/security/xray/pkg/checks/check"
	"a.yandex-team.ru/security/xray/pkg/xrayrpc"
)

const (
	Name = "Soxer"
	Type = "soxer"
)

var _ check.StageCheck = (*Soxer)(nil)
var _ check.Check = (*Soxer)(nil)
var issueID = base64.RawURLEncoding.EncodeToString([]byte(Type))

type (
	Soxer struct {
		abcClient *abc.Client
	}
)

func New(env check.Config) *Soxer {
	abcClient, err := abc.NewClient(env.AuthToken)
	if err != nil {
		panic(fmt.Sprintf("failed to create ABC client: %s", err))
	}

	return &Soxer{
		abcClient: abcClient,
	}
}

func (s *Soxer) Name() string {
	return Name
}

func (s *Soxer) Type() string {
	return Type
}

func (s *Soxer) Sync(_ context.Context) error {
	return nil
}

func (s *Soxer) Deadline() time.Duration {
	return 1 * time.Minute
}

func (s *Soxer) CheckStage(l log.Logger, stage *check.StageSpec) (check.Issues, error) {
	if stage.Stage.Spec.SoxService != nil {
		// OK
		return nil, nil
	}

	accountID := stage.Stage.Spec.GetAccountId()
	abcID := yputil.AccountToABC(accountID)
	if abcID == 0 {
		l.Error("failed to parse account ID", log.String("account_id", accountID))
		// TODO(buglloc): return warnings
		return nil, nil
	}

	tags, err := s.abcClient.GetServiceTags(context.Background(), abcID)
	if err != nil {
		l.Error("failed to get ABC account tags", log.Int("abc_id", abcID), log.Error(err))
		// TODO(buglloc): return warnings
		return nil, nil
	}

	if !abc.IsSoxTags(tags) {
		// not a sox service
		return nil, nil
	}

	return []*xrayrpc.Issue{{
		Id:       issueID,
		Kind:     xrayrpc.IssueKind_IK_SECURITY,
		Severity: xrayrpc.Severity_S_MEDIUM,
		Details: &xrayrpc.Issue_Soxer{
			Soxer: &xrayrpc.SoxerIssueDetail{
				AbcId:   uint32(abcID),
				AbcTags: abc.CompactTags(tags),
			},
		},
	}}, nil
}
