package spring4shellfinder

import (
	"bytes"
	"context"
	"encoding/base64"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/xray/internal/spring4shell-finder/vuln"
	"a.yandex-team.ru/security/xray/pkg/checks/check"
	"a.yandex-team.ru/security/xray/pkg/collectors/spring4shellfinder"
	"a.yandex-team.ru/security/xray/pkg/xrayrpc"
)

const (
	Name = "Spring4Shell finder"
	Type = "spring4shell-finder"
)

var (
	_          check.BoxFSCheck = (*Boxer)(nil)
	_          check.Check      = (*Boxer)(nil)
	collectors                  = []string{spring4shellfinder.Type}
)

type (
	Boxer struct {
		command string
	}
)

func New(_ check.Config) *Boxer {
	return &Boxer{}
}

func (s *Boxer) Name() string {
	return Name
}

func (s *Boxer) Type() string {
	return Type
}

func (s *Boxer) Command() string {
	return s.command
}

func (s *Boxer) Collectors() []string {
	return collectors
}

func (s *Boxer) Sync(_ context.Context) error {
	return nil
}

func (s *Boxer) Deadline() time.Duration {
	return 5 * time.Minute
}

func (s *Boxer) ProcessCollectorResults(_ log.Logger, result map[string]*xrayrpc.CollectorResult) (check.Issues, error) {
	var badJars []*xrayrpc.Spring4ShellFinderIssueDetail_Spring4ShellInfo
	spring4ShellFinderInfo := result[spring4shellfinder.Type].Finding.GetSpring4ShellFinder()

	if spring4ShellFinderInfo == nil {
		return nil, nil
	}

	for _, jar := range spring4ShellFinderInfo.Jars {
		if !vuln.IsVulnerable(jar) {
			continue
		}

		badJars = append(badJars, &xrayrpc.Spring4ShellFinderIssueDetail_Spring4ShellInfo{
			Path:    jar.Path,
			Type:    jar.Type,
			Name:    jar.Name,
			Version: jar.Version,
		})
	}

	if len(badJars) == 0 {
		return nil, nil
	}

	return []*xrayrpc.Issue{{
		Id:       generateIssueID(&xrayrpc.Spring4ShellFinderIssueDetail{}),
		Severity: xrayrpc.Severity_S_HIGH,
		Kind:     xrayrpc.IssueKind_IK_SECURITY,
		Details: &xrayrpc.Issue_Spring4ShellFinder{
			Spring4ShellFinder: &xrayrpc.Spring4ShellFinderIssueDetail{
				Jars: badJars,
			},
		},
	}}, nil
}

func generateIssueID(_ *xrayrpc.Spring4ShellFinderIssueDetail) string {
	var buf bytes.Buffer
	buf.WriteString(Type)

	return base64.RawURLEncoding.EncodeToString(buf.Bytes())
}
