package yadios

import (
	"bytes"
	"context"
	"crypto/tls"
	"encoding/base64"
	"errors"
	"fmt"
	"net/http"
	"os"
	"path/filepath"
	"strings"
	"time"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/certifi"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/libs/go/ioatomic"
	"a.yandex-team.ru/security/libs/go/lineage"
	"a.yandex-team.ru/security/xray/pkg/checks/check"
	"a.yandex-team.ru/security/xray/pkg/collectors/boxer"
	"a.yandex-team.ru/security/xray/pkg/collectors/yadios"
	"a.yandex-team.ru/security/xray/pkg/xrayrpc"
	"a.yandex-team.ru/security/yadi/yadi-os/pkg/analyze"
	"a.yandex-team.ru/security/yadi/yadi-os/pkg/debversion"
	"a.yandex-team.ru/security/yadi/yadi-os/pkg/pkgmanager/apk"
	"a.yandex-team.ru/security/yadi/yadi-os/pkg/pkgmanager/dpkg"
	"a.yandex-team.ru/security/yadi/yadi-os/pkg/pkgmanager/manager"
	"a.yandex-team.ru/security/yadi/yadi-os/pkg/pkgmanager/mockmngr"
)

const (
	Name = "YadiOS"
	Type = "yadi-os"
)

var (
	_          check.BoxFSCheck = (*YadiOS)(nil)
	_          check.Check      = (*YadiOS)(nil)
	collectors                  = []string{boxer.Type, yadios.Type}
)

type (
	YadiOS struct {
		dbPath string
		httpc  *resty.Client
	}

	listYadiFiles struct {
		Files []string `json:"files"`
	}
)

func New(env check.Config) *YadiOS {
	certPool, err := certifi.NewCertPool()
	if err != nil {
		panic(err)
	}

	dbPath := filepath.Join(env.HostDir, "db", "yadi")
	err = os.MkdirAll(dbPath, 0755)
	if err != nil {
		panic(err)
	}

	httpc := resty.New().
		SetBaseURL("https://yadi.yandex-team.ru").
		SetRedirectPolicy(resty.NoRedirectPolicy()).
		SetTLSClientConfig(&tls.Config{RootCAs: certPool})

	return &YadiOS{
		dbPath: dbPath,
		httpc:  httpc,
	}
}

func (s *YadiOS) Name() string {
	return Name
}

func (s *YadiOS) Type() string {
	return Type
}

func (s *YadiOS) Collectors() []string {
	return collectors
}

func (s *YadiOS) Sync(ctx context.Context) error {
	files, err := listFiles(ctx, s.httpc)
	if err != nil {
		return err
	}

	if len(files) == 0 {
		return errors.New("yadi manifest.json have no acceptable files")
	}

	for _, f := range files {
		err := func() error {
			res, err := s.httpc.R().
				SetContext(ctx).
				SetDoNotParseResponse(true).
				Get("/db/" + f)

			if err != nil {
				return fmt.Errorf("failed to get db file %q: %w", f, err)
			}

			body := res.RawBody()
			defer func() { _ = body.Close() }()

			if !res.IsSuccess() {
				return fmt.Errorf("failed to get db file %q: non-200 status code: %d", f, res.StatusCode())
			}

			dbFilePath := filepath.Join(s.dbPath, f)
			err = ioatomic.WriteFile(dbFilePath, res.RawBody(), 0644)
			if err != nil {
				return fmt.Errorf("failed to save file %q: %w", dbFilePath, err)
			}
			return nil
		}()

		if err != nil {
			return err
		}
	}
	return nil
}

func (s *YadiOS) Deadline() time.Duration {
	return 5 * time.Minute
}

func (s *YadiOS) ProcessCollectorResults(logger log.Logger, result map[string]*xrayrpc.CollectorResult) (check.Issues, error) {
	boxerInfo := result[boxer.Type].Finding.GetBoxer()
	yadiosInfo := result[yadios.Type].Finding.GetYadiOs()

	if boxerInfo == nil || yadiosInfo == nil {
		return nil, nil
	}

	packages := make([]*manager.Package, 0, len(yadiosInfo.Packages))
	for _, pkg := range yadiosInfo.Packages {
		if pkg.Version == "" {
			logger.Warn("ignore pkg with empty version",
				log.String("package_name", pkg.Name))
			continue
		}

		debVersion, err := debversion.NewVersion(pkg.Version)
		if err != nil {
			logger.Warn("failed to convert to deb-version",
				log.String("package_name", pkg.Name),
				log.String("package_version", pkg.Version))
			continue
		}

		// TODO(buglloc): reimplement version + source version relation
		var debSourceVersion *debversion.Version
		if pkg.SourceVersion != "" {
			debSourceVersion, err = debversion.NewVersion(pkg.SourceVersion)
			if err != nil {
				logger.Warn("failed to convert to deb-version",
					log.String("package_name", pkg.Name),
					log.String("package_source_version", pkg.SourceVersion))
				continue
			}
		} else {
			debSourceVersion = debVersion
		}

		packages = append(packages, &manager.Package{
			Name:             pkg.Name,
			Version:          debVersion,
			RawVersion:       pkg.Version,
			SourceName:       pkg.SourceName,
			SourceVersion:    debSourceVersion,
			RawSourceVersion: pkg.SourceVersion,
		})
	}

	analyzer, err := analyze.New(analyze.Options{
		FeedURI:         filepath.Join(s.dbPath, "linux-{ecosystem}.json"),
		MinimumSeverity: 6,
		FixableOnly:     true,
	})
	if err != nil {
		return nil, err
	}

	pkgManager := mockmngr.NewManager(mockmngr.ManagerInfo{
		Name:                 "x-ray",
		Distributive:         strings.ToLower(boxerInfo.Distro.Family),
		DistributiveCodename: strings.ToLower(boxerInfo.Distro.Codename),
		Packages:             packages,
	})

	checkResults, err := analyzer.CheckPackages(pkgManager)
	if err != nil {
		return nil, err
	}

	if len(checkResults) <= 0 {
		return nil, nil
	}

	issues := make([]*xrayrpc.Issue, len(checkResults))
	for i, pkg := range checkResults {
		issues[i] = &xrayrpc.Issue{
			Id:       generateIssueID(pkg.Package.Name),
			Kind:     xrayrpc.IssueKind_IK_SECURITY,
			Severity: xraySeverity(maxScore(pkg.Vulnerabilities)),
			Details: &xrayrpc.Issue_YadiOs{
				YadiOs: &xrayrpc.YadiOsIssueDetail{
					Package: &xrayrpc.YadiOsIssueDetail_Package{
						Name:          pkg.Package.Name,
						Version:       pkg.Package.Version.String(),
						SourceName:    pkg.Package.SourceName,
						SourceVersion: pkg.Package.SourceVersion.String(),
					},
					Vulnerabilities: toXrayVulnerabilities(pkg.Vulnerabilities),
				},
			},
		}
	}

	return issues, nil
}

func (s *YadiOS) detectPkgManager(currentOS *lineage.OS) (manager.Manager, error) {
	// FIXME(anton-k): we don't need Root here
	switch currentOS.Family {
	case lineage.Ubuntu, lineage.Debian:
		return dpkg.NewManager(dpkg.ManagerOpts{
			OS:   currentOS,
			Root: "/",
		})
	case lineage.Alpine:
		return apk.NewManager(apk.ManagerOpts{
			OS:   currentOS,
			Root: "/",
		})
	default:
		return nil, errors.New("unsupported OS family")
	}
}

func xraySeverity(cvssScore float32) xrayrpc.Severity {
	switch {
	case cvssScore <= 3.9:
		return xrayrpc.Severity_S_INFO
	case cvssScore <= 6.9:
		return xrayrpc.Severity_S_LOW
	case cvssScore <= 8.9:
		return xrayrpc.Severity_S_MEDIUM
	default:
		// temporary filter out HIGH severity status
		// https://st.yandex-team.ru/XRAY-47
		// return xrayrpc.Severity_S_HIGH
		return xrayrpc.Severity_S_MEDIUM
	}
}

func listFiles(ctx context.Context, httpc *resty.Client) ([]string, error) {
	var index listYadiFiles
	rsp, err := httpc.R().
		SetResult(&index).
		SetContext(ctx).
		Get("/db/manifest.json")

	if err != nil {
		return nil, err
	}

	if rsp.StatusCode() != http.StatusOK {
		return nil, fmt.Errorf("failed to list yadi files, not 200 status code: %d", rsp.StatusCode())
	}

	var out []string
	for _, f := range index.Files {
		if !strings.HasSuffix(f, ".json") || f == "web.json" {
			continue
		}

		out = append(out, f)
	}
	return out, nil
}

func maxScore(issues analyze.Issues) float32 {
	score := float32(0.0)
	for _, issue := range issues {
		if issue.CVSSScore > score {
			score = issue.CVSSScore
		}
	}
	return score
}

func toXrayVulnerabilities(issues analyze.Issues) []*xrayrpc.YadiOsIssueDetail_Vulnerability {
	vulns := make([]*xrayrpc.YadiOsIssueDetail_Vulnerability, len(issues))
	for i, issue := range issues {
		vulns[i] = &xrayrpc.YadiOsIssueDetail_Vulnerability{
			Id:               issue.ID,
			AffectedVersions: issue.AffectedVersions,
			Reference:        issue.Reference,
			Summary:          issue.Summary,
			CvssScore:        issue.CVSSScore,
		}
	}
	return vulns
}

func generateIssueID(pkgName string) string {
	var buf bytes.Buffer
	buf.WriteString(Type)
	buf.WriteByte(':')
	buf.WriteString(pkgName)

	return base64.RawURLEncoding.EncodeToString(buf.Bytes())
}
