package yadios

import (
	"bytes"
	"fmt"
	"strings"
	"text/template"

	"a.yandex-team.ru/security/xray/pkg/checks/check"
	"a.yandex-team.ru/security/xray/pkg/xrayrpc"
	"a.yandex-team.ru/security/yadi/libs/cvs"
)

const (
	aboutURL = "https://wiki.yandex-team.ru/security/x-ray/checks/#yadios"
	summary  = `The %q package has known vulnerabilities`
	maxVulns = 5
)

type (
	tmplIssueDetail struct {
		Package         *xrayrpc.YadiOsIssueDetail_Package
		Vulnerabilities []*xrayrpc.YadiOsIssueDetail_Vulnerability
		RestCount       int
	}
)

var (
	description, _ = template.New("description").
			Funcs(templateFns).
			Parse(`
Package info:
  - package name: {{.Package.Name}}
  - package version: {{.Package.Version}}
{{if .Package.SourceName}}  - source package: {{.Package.SourceName}}
{{if .Package.SourceVersion}}  - source version: {{.Package.SourceVersion}}{{end}}{{end}}

{{range $i, $vulnerability := .Vulnerabilities}}
{{CvssScoreToSeverity $vulnerability.CvssScore}} severity vulnerability [{{$vulnerability.Id}}]({{$vulnerability.Reference}}):
  - summary: {{$vulnerability.Summary}}
  - affected versions: ` + "`" + `{{$vulnerability.AffectedVersions}}` + "`" + `

{{end}}
{{if .RestCount}}And {{.RestCount}} more{{end}}
`)

	templateFns = template.FuncMap{
		"CvssScoreToSeverity": cvs.ToSeverity,
	}
)

func FormatIssue(issue *xrayrpc.Issue) (*check.FormattedIssue, error) {
	details, ok := issue.Details.(*xrayrpc.Issue_YadiOs)
	if !ok {
		return nil, check.ErrWrongIssueType
	}

	max := maxVulns
	if max > len(details.YadiOs.Vulnerabilities) {
		max = len(details.YadiOs.Vulnerabilities)
	}

	tmplData := &tmplIssueDetail{
		Package:         details.YadiOs.Package,
		Vulnerabilities: details.YadiOs.Vulnerabilities[:max],
		RestCount:       len(details.YadiOs.Vulnerabilities) - max,
	}

	var buf bytes.Buffer
	err := description.Execute(&buf, tmplData)
	if err != nil {
		return nil, fmt.Errorf("failed to format description: %w", err)
	}

	return &check.FormattedIssue{
		ID:          issue.Id,
		Summary:     fmt.Sprintf(summary, details.YadiOs.GetPackage().GetName()),
		HelpURL:     aboutURL,
		Description: strings.TrimSpace(buf.String()),
	}, nil
}
