package yodax

import (
	"bytes"
	"context"
	"encoding/base64"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/xray/pkg/checks/check"
	"a.yandex-team.ru/security/xray/pkg/collectors/yodax"
	"a.yandex-team.ru/security/xray/pkg/xrayrpc"
)

const (
	Name = "Yodax"
	Type = "yodax"
)

var (
	_          check.BoxFSCheck = (*Yodax)(nil)
	_          check.Check      = (*Yodax)(nil)
	collectors                  = []string{yodax.Type}
)

type (
	Yodax struct{}
)

func New(_ check.Config) *Yodax {
	return &Yodax{}
}

func (s *Yodax) Name() string {
	return Name
}

func (s *Yodax) Type() string {
	return Type
}

func (s *Yodax) Collectors() []string {
	return collectors
}

func (s *Yodax) Sync(_ context.Context) error {
	return nil
}

func (s *Yodax) Deadline() time.Duration {
	return 5 * time.Minute
}

func (s *Yodax) ProcessCollectorResults(_ log.Logger, result map[string]*xrayrpc.CollectorResult) (issues check.Issues, err error) {
	yodaxDetails := result[yodax.Type].Finding.GetYodax()
	if yodaxDetails == nil {
		return nil, nil
	}

	for _, yodaxInfo := range yodaxDetails.Problems {
		issues = append(issues, &xrayrpc.Issue{
			Id:       generateIssueID(yodaxInfo),
			Kind:     xrayrpc.IssueKind_IK_SECURITY,
			Severity: parseSeverity(yodaxInfo.Severity),
			Details: &xrayrpc.Issue_Yodax{
				Yodax: &xrayrpc.YodaxIssueDetail{
					Path:        yodaxInfo.Path,
					Config:      yodaxInfo.Config,
					Description: yodaxInfo.Description,
					Plugin:      yodaxInfo.Plugin,
					Reason:      yodaxInfo.Reason,
					Reference:   yodaxInfo.Reference,
					Summary:     yodaxInfo.Summary,
				},
			},
		})

	}
	return
}

func parseSeverity(severity xrayrpc.YodaxFindingDetail_YodaxSeverity) xrayrpc.Severity {
	switch severity {
	case xrayrpc.YodaxFindingDetail_YS_HIGH:
		return xrayrpc.Severity_S_HIGH
	case xrayrpc.YodaxFindingDetail_YS_MEDIUM:
		return xrayrpc.Severity_S_MEDIUM
	case xrayrpc.YodaxFindingDetail_YS_LOW:
		return xrayrpc.Severity_S_LOW
	case xrayrpc.YodaxFindingDetail_YS_UNSPECIFIED:
		fallthrough
	default:
		return xrayrpc.Severity_S_INFO
	}
}

func generateIssueID(finding *xrayrpc.YodaxFindingDetail_YodaxInfo) string {
	var buf bytes.Buffer
	buf.WriteString(Type)
	buf.WriteByte(':')
	buf.WriteString(finding.Plugin)
	buf.WriteByte(';')
	buf.WriteString(finding.Path)

	return base64.RawURLEncoding.EncodeToString(buf.Bytes())
}
