package boxer

import (
	"bytes"
	"context"
	"encoding/base64"
	"fmt"
	"time"

	"google.golang.org/protobuf/proto"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/xray/internal/boxer/boxer_out"
	"a.yandex-team.ru/security/xray/pkg/collectors/collector"
	"a.yandex-team.ru/security/xray/pkg/xrayrpc"
)

const (
	Name    = "Boxer"
	Type    = "boxer"
	Version = "0.1"
)

var (
	_ collector.BoxFSCollector = (*Boxer)(nil)
	_ collector.Collector      = (*Boxer)(nil)
)

type (
	Boxer struct {
		command string
	}
)

func New(env collector.Config) *Boxer {
	return &Boxer{
		command: fmt.Sprintf("%s/boxer", env.ContainerDir),
	}
}

func (s *Boxer) Name() string {
	return Name
}

func (s *Boxer) Type() string {
	return Type
}

func (s *Boxer) Version() string {
	return Version
}

func (s *Boxer) Sync(_ context.Context) error {
	return nil
}

func (s *Boxer) Command() string {
	return s.command
}

func (s *Boxer) Requirements() collector.ContainerRequirements {
	return collector.ContainerRequirements{
		Net: false,
	}
}

func (s *Boxer) Deadline() time.Duration {
	return 1 * time.Minute
}

func (s *Boxer) ProcessContainerResult(_ log.Logger, result *collector.ContainerResult, fsID string) (*xrayrpc.Finding, error) {
	exitStatus := result.Status.ExitStatus()
	switch exitStatus {
	case 0:
		// ok
	default:
		return nil, fmt.Errorf("unexpected box-info exit code %d. stderr: %s", exitStatus, string(result.Stderr))
	}

	if len(result.Stdout) <= 0 {
		// that's ok
		return nil, nil
	}

	var boxerOut boxer_out.CollectorResult
	if err := proto.Unmarshal(result.Stdout, &boxerOut); err != nil {
		return nil, fmt.Errorf("failed to decode box-info results: %w", err)
	}

	return &xrayrpc.Finding{
		Id: generateFindingID(fsID),
		Details: &xrayrpc.Finding_Boxer{
			Boxer: boxerOut.Finding,
		},
	}, nil
}

func generateFindingID(fsID string) string {
	var buf bytes.Buffer
	buf.WriteString(Type)
	buf.WriteByte(':')
	buf.WriteString(fsID)

	return base64.RawURLEncoding.EncodeToString(buf.Bytes())
}
