package collector

import (
	"context"
	"syscall"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/xray/pkg/xrayrpc"
)

type (
	TargetKind int32
)

const (
	TargetKindUnknown TargetKind = iota
	TargetKindBoxFS
)

type (
	Collector interface {
		// Human-readable collector name
		Name() string

		// Collector type
		Type() string

		// Collector version
		Version() string

		// Maximum collector time
		Deadline() time.Duration

		// Periodic sync db data
		Sync(ctx context.Context) error
	}

	BoxFSCollector interface {
		Collector
		// Must return command to execude inside container
		// E.g.: cat /etc/lsb-release
		Command() string

		// BoxFSCheck requirements
		Requirements() ContainerRequirements

		// ProcessContainerResult results
		ProcessContainerResult(logger log.Logger, result *ContainerResult, fsID string) (*xrayrpc.Finding, error)
	}

	ContainerRequirements struct {
		// Require network
		Net bool

		// Require bind mounts
		// E.g.: /home/tmp /tmp
		Bind []string

		// Environments variables
		Env []string
	}

	ContainerResult struct {
		Status  syscall.WaitStatus
		Stdout  []byte
		Stderr  []byte
		Elapsed time.Duration
	}

	Config struct {
		ContainerDir string
		HostDir      string
	}
)

func (c *ContainerRequirements) PortoNet() string {
	if c.Net {
		// TODO(buglloc): ooops!
		return "host"
	}
	return "none"
}

func MergeContainerRequirements(a, b ContainerRequirements) ContainerRequirements {
	a.Net = a.Net || b.Net
	a.Env = append(a.Env, b.Env...)
	a.Bind = append(a.Bind, b.Bind...)
	return a
}
