package log4jfinder

import (
	"bytes"
	"context"
	"encoding/base64"
	"fmt"
	"time"

	"google.golang.org/protobuf/proto"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/xray/internal/log4j-finder/log4jfinder_out"
	"a.yandex-team.ru/security/xray/pkg/collectors/collector"
	"a.yandex-team.ru/security/xray/pkg/xrayrpc"
)

const (
	Name    = "Log4j finder"
	Type    = "log4j-finder"
	Version = "0.1"
)

var (
	_ collector.BoxFSCollector = (*Boxer)(nil)
	_ collector.Collector      = (*Boxer)(nil)
)

type (
	Boxer struct {
		command string
	}
)

func New(env collector.Config) *Boxer {
	return &Boxer{
		command: fmt.Sprintf("%s/log4j-finder --collect-only", env.ContainerDir),
	}
}

func (s *Boxer) Name() string {
	return Name
}

func (s *Boxer) Type() string {
	return Type
}

func (s *Boxer) Version() string {
	return Version
}

func (s *Boxer) Command() string {
	return s.command
}

func (s *Boxer) Sync(_ context.Context) error {
	return nil
}

func (s *Boxer) Requirements() collector.ContainerRequirements {
	return collector.ContainerRequirements{
		Net: false,
	}
}

func (s *Boxer) Deadline() time.Duration {
	return 5 * time.Minute
}

func (s *Boxer) ProcessContainerResult(_ log.Logger, result *collector.ContainerResult, fsID string) (*xrayrpc.Finding, error) {
	exitStatus := result.Status.ExitStatus()
	switch exitStatus {
	case 0:
		// ok
	default:
		return nil, fmt.Errorf("unexpected log4j-finder exit code %d. stderr: %s", exitStatus, string(result.Stderr))
	}

	if len(result.Stdout) <= 0 {
		// that's ok
		return nil, nil
	}

	var checkOut log4jfinder_out.Result
	if err := proto.Unmarshal(result.Stdout, &checkOut); err != nil {
		return nil, fmt.Errorf("failed to decode box-info results: %w", err)
	}

	jars := make([]*xrayrpc.Log4JFinderFindingDetail_Log4JInfo, len(checkOut.Issue.Jars))
	for i := range jars {
		jars[i] = &xrayrpc.Log4JFinderFindingDetail_Log4JInfo{
			Path:    checkOut.Issue.Jars[i].Path,
			Version: checkOut.Issue.Jars[i].Version,
		}
	}

	return &xrayrpc.Finding{
		Id: generateFindingID(fsID),
		Details: &xrayrpc.Finding_Log4JFinder{
			Log4JFinder: &xrayrpc.Log4JFinderFindingDetail{Jars: jars},
		},
	}, nil
}

func generateFindingID(fsID string) string {
	var buf bytes.Buffer
	buf.WriteString(Type)
	buf.WriteByte(':')
	buf.WriteString(fsID)

	return base64.RawURLEncoding.EncodeToString(buf.Bytes())
}
