package secretsearch

import (
	"bytes"
	"context"
	"encoding/base64"
	"fmt"
	"net"
	"path/filepath"
	"strings"
	"time"

	"google.golang.org/protobuf/proto"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/ant-secret/secret-search/public/cpp/output/secret_search_out"
	"a.yandex-team.ru/security/xray/internal/stringutil"
	"a.yandex-team.ru/security/xray/pkg/collectors/collector"
	"a.yandex-team.ru/security/xray/pkg/xrayrpc"
)

const (
	Name    = "Secrets searcher"
	Type    = "secret-search"
	Version = "0.1"

	antsecretHost = "ant.sec.yandex-team.ru:80"
)

var _ collector.BoxFSCollector = (*SecretSearch)(nil)
var _ collector.Collector = (*SecretSearch)(nil)

type (
	SecretSearch struct {
		containerCommand string
		endpointAddr     string
	}
)

var (
	excludes = []string{
		".arc",
		".git",
		".hg",
		".svn",
		".venv",
		".venv3",
		"node_modules",
		"tmp",
		"dev",
		"proc",
		"sys",
		"run",
		"berkanavt",
		"_cacache",
	}
)

func New(env collector.Config) *SecretSearch {
	return &SecretSearch{
		containerCommand: strings.Join([]string{
			filepath.Join(env.ContainerDir, "secret-search"),
			"--status-code=3",
			"--format=proto",
			"--num-threads=1",
			"--valid-only",
			"--excludes=" + strings.Join(excludes, ","),
			"/",
		}, " "),
	}
}

func (s *SecretSearch) Name() string {
	return Name
}

func (s *SecretSearch) Type() string {
	return Type
}

func (s *SecretSearch) Version() string {
	return Version
}

func (s *SecretSearch) Command() string {
	return s.containerCommand
}

func (s *SecretSearch) Sync(ctx context.Context) error {
	addr, err := net.ResolveTCPAddr("tcp6", antsecretHost)
	if err != nil {
		return fmt.Errorf("failed to resolve ant-secret host %q: %w", antsecretHost, err)
	}

	s.endpointAddr = "http://[" + addr.IP.String() + "]:80/api/v1/validate/"
	return nil
}

func (s *SecretSearch) Requirements() collector.ContainerRequirements {
	return collector.ContainerRequirements{
		Net: true,
		Env: []string{"ANT_VALIDATION_ENDPOINT=" + s.endpointAddr},
	}
}

func (s *SecretSearch) Deadline() time.Duration {
	return 2 * time.Minute
}

func (s *SecretSearch) ProcessContainerResult(logger log.Logger, result *collector.ContainerResult, fsID string) (*xrayrpc.Finding, error) {
	exitStatus := result.Status.ExitStatus()
	switch exitStatus {
	case 0:
		// nothing to do
		return nil, nil
	case 3:
		// ok
	default:
		return nil, fmt.Errorf("unexpected secret-search exit code %d", exitStatus)
	}
	findings, err := parseOut(result.Stdout, xrayrpc.SecretSearchFindingDetail_SSIK_BOX, fsID)
	if err != nil {
		logger.Error("unexpected secret-search result", log.String("stdout", string(result.Stdout)))
		return nil, err
	}

	return findings, nil
}

func parseOut(out []byte, kind xrayrpc.SecretSearchFindingDetail_Kind, fsID string) (*xrayrpc.Finding, error) {
	if len(out) == 0 {
		return nil, nil
	}

	var ssOut secret_search_out.Result
	if err := proto.Unmarshal(out, &ssOut); err != nil {
		return nil, fmt.Errorf("failed to decode secret-search results: %w", err)
	}

	if len(ssOut.Issues) <= 0 {
		// ok
		return nil, nil
	}

	findings := make([]*xrayrpc.SecretSearchFindingDetail_SecretInfo, 0, len(ssOut.Issues))
	for _, issue := range ssOut.Issues {
		for _, secret := range issue.Secrets {
			secretHash, ok := secret.Additional["sha1"]
			if !ok {
				secretHash = stringutil.ShaHex(secret.Secret)
			}

			findings = append(findings, &xrayrpc.SecretSearchFindingDetail_SecretInfo{
				Kind:      kind,
				Path:      issue.Path,
				Type:      secret.Type,
				Validated: secret.Validated,
				LineNo:    secret.LineNo,
				Secret:    secret.Secret,
				Hash:      secretHash,
			})
		}
	}
	return &xrayrpc.Finding{
		Id: generateFindingID(kind, fsID),
		Details: &xrayrpc.Finding_SecretSearch{
			SecretSearch: &xrayrpc.SecretSearchFindingDetail{
				SecretsInfo: findings,
			},
		},
	}, nil
}

func generateFindingID(kind xrayrpc.SecretSearchFindingDetail_Kind, fsID string) string {
	var buf bytes.Buffer
	buf.WriteString(Type)
	buf.WriteByte(':')
	buf.WriteString(kind.String())
	buf.WriteByte(':')
	buf.WriteString(fsID)

	return base64.RawURLEncoding.EncodeToString(buf.Bytes())
}
