package spring4shellfinder

import (
	"bytes"
	"context"
	"encoding/base64"
	"fmt"
	"time"

	"google.golang.org/protobuf/proto"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/xray/internal/spring4shell-finder/spring4shellfinder_out"
	"a.yandex-team.ru/security/xray/pkg/collectors/collector"
	"a.yandex-team.ru/security/xray/pkg/xrayrpc"
)

const (
	Name    = "Spring4Shell finder"
	Type    = "spring4shell-finder"
	Version = "0.2"
)

var (
	_ collector.BoxFSCollector = (*Boxer)(nil)
	_ collector.Collector      = (*Boxer)(nil)
)

type (
	Boxer struct {
		command string
	}
)

func New(env collector.Config) *Boxer {
	return &Boxer{
		command: fmt.Sprintf("%s/spring4shell-finder --collect-only", env.ContainerDir),
	}
}

func (s *Boxer) Name() string {
	return Name
}

func (s *Boxer) Type() string {
	return Type
}

func (s *Boxer) Version() string {
	return Version
}

func (s *Boxer) Command() string {
	return s.command
}

func (s *Boxer) Sync(_ context.Context) error {
	return nil
}

func (s *Boxer) Requirements() collector.ContainerRequirements {
	return collector.ContainerRequirements{
		Net: false,
	}
}

func (s *Boxer) Deadline() time.Duration {
	return 5 * time.Minute
}

func (s *Boxer) ProcessContainerResult(_ log.Logger, result *collector.ContainerResult, fsID string) (*xrayrpc.Finding, error) {
	exitStatus := result.Status.ExitStatus()
	switch exitStatus {
	case 0:
		// ok
	default:
		return nil, fmt.Errorf("unexpected spring4shell-finder exit code %d. stderr: %s", exitStatus, string(result.Stderr))
	}

	if len(result.Stdout) <= 0 {
		// that's ok
		return nil, nil
	}

	var checkOut spring4shellfinder_out.Result
	if err := proto.Unmarshal(result.Stdout, &checkOut); err != nil {
		return nil, fmt.Errorf("failed to decode box-info results: %w", err)
	}

	return &xrayrpc.Finding{
		Id: generateFindingID(fsID),
		Details: &xrayrpc.Finding_Spring4ShellFinder{
			Spring4ShellFinder: &xrayrpc.Spring4ShellFinderFindingDetail{
				Jars: checkOut.Issue.Jars,
			},
		},
	}, nil
}

func generateFindingID(fsID string) string {
	var buf bytes.Buffer
	buf.WriteString(Type)
	buf.WriteByte(':')
	buf.WriteString(fsID)

	return base64.RawURLEncoding.EncodeToString(buf.Bytes())
}
