package yadios

import (
	"bytes"
	"context"
	"crypto/tls"
	"encoding/base64"
	"fmt"
	"net/http"
	"os"
	"path/filepath"
	"strings"
	"time"

	"github.com/go-resty/resty/v2"
	"google.golang.org/protobuf/proto"

	"a.yandex-team.ru/library/go/certifi"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/xray/pkg/collectors/collector"
	"a.yandex-team.ru/security/xray/pkg/xrayrpc"
	"a.yandex-team.ru/security/yadi/yadi-os/pkg/formatter/protofmt/yadi_os_out"
)

const (
	Name    = "YadiOS"
	Type    = "yadi-os"
	Version = "0.1"
)

var (
	_ collector.BoxFSCollector = (*YadiOS)(nil)
	_ collector.Collector      = (*YadiOS)(nil)
)

type (
	YadiOS struct {
		command     string
		dbPath      string
		innerDBPath string
		httpc       *resty.Client
	}

	listYadiFiles struct {
		Files []string `json:"files"`
	}
)

func New(env collector.Config) *YadiOS {
	certPool, err := certifi.NewCertPool()
	if err != nil {
		panic(err)
	}

	dbPath := filepath.Join(env.HostDir, "db", "yadi")
	err = os.MkdirAll(dbPath, 0755)
	if err != nil {
		panic(err)
	}

	httpc := resty.New().
		SetBaseURL("https://yadi.yandex-team.ru").
		SetRedirectPolicy(resty.NoRedirectPolicy()).
		SetTLSClientConfig(&tls.Config{RootCAs: certPool})

	return &YadiOS{
		// need yadi-os info?
		command: strings.Join([]string{
			filepath.Join(env.ContainerDir, "yadi-os"),
			"list",
			"--format=proto",
			"--skip-unknown-os",
		}, " "),
		dbPath:      dbPath,
		innerDBPath: filepath.Join(env.ContainerDir, "db", "yadi", "linux-{ecosystem}.json"),
		httpc:       httpc,
	}
}

func (s *YadiOS) Name() string {
	return Name
}

func (s *YadiOS) Type() string {
	return Type
}

func (s *YadiOS) Command() string {
	return s.command
}

func (s *YadiOS) Version() string {
	return Version
}

func (s *YadiOS) Sync(ctx context.Context) error {
	return nil
}

func (s *YadiOS) Requirements() collector.ContainerRequirements {
	return collector.ContainerRequirements{
		Net: false,
		Env: []string{"YADIOS_FEED=" + s.innerDBPath},
	}
}

func (s *YadiOS) Deadline() time.Duration {
	return 5 * time.Minute
}

func (s *YadiOS) ProcessContainerResult(logger log.Logger, result *collector.ContainerResult, fsID string) (finding *xrayrpc.Finding, err error) {
	exitStatus := result.Status.ExitStatus()
	switch exitStatus {
	case 0:
		// ok
	case 3:
		// ok
	default:
		err = fmt.Errorf("unexpected yadi-os exit code %d. stderr: %s", exitStatus, string(result.Stderr))
		return
	}

	if len(result.Stdout) <= 0 {
		// that's ok
		return
	}

	var yadiOut yadi_os_out.ListResult
	err = proto.Unmarshal(result.Stdout, &yadiOut)
	if err != nil {
		err = fmt.Errorf("failed to decode yadi-os results: %w", err)
		logger.Error("unexpected yadi-os result", log.String("stdout", string(result.Stdout)))
		return
	}

	if len(yadiOut.Packages) <= 0 {
		return
	}

	packages := make([]*xrayrpc.YadiOsFindingDetail_Package, len(yadiOut.Packages))
	for i, pkg := range yadiOut.Packages {
		packages[i] = &xrayrpc.YadiOsFindingDetail_Package{
			Name:          pkg.Name,
			Version:       pkg.Version,
			SourceName:    pkg.SourceName,
			SourceVersion: pkg.SourceVersion,
		}
	}
	return &xrayrpc.Finding{
		Id: generateFindingID(fsID),
		Details: &xrayrpc.Finding_YadiOs{
			YadiOs: &xrayrpc.YadiOsFindingDetail{Packages: packages},
		},
	}, nil
}

func listFiles(ctx context.Context, httpc *resty.Client) ([]string, error) {
	var index listYadiFiles
	rsp, err := httpc.R().
		SetResult(&index).
		SetContext(ctx).
		Get("/db/manifest.json")

	if err != nil {
		return nil, err
	}

	if rsp.StatusCode() != http.StatusOK {
		return nil, fmt.Errorf("failed to list yadi files, not 200 status code: %d", rsp.StatusCode())
	}

	var out []string
	for _, f := range index.Files {
		if !strings.HasSuffix(f, ".json") || f == "web.json" {
			continue
		}

		out = append(out, f)
	}
	return out, nil
}

func generateFindingID(fsID string) string {
	var buf bytes.Buffer
	buf.WriteString(Type)
	buf.WriteByte(':')
	buf.WriteString(fsID)

	return base64.RawURLEncoding.EncodeToString(buf.Bytes())
}
