package yodax

import (
	"bytes"
	"context"
	"encoding/base64"
	"fmt"
	"time"

	"google.golang.org/protobuf/proto"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/security/xray/pkg/collectors/collector"
	"a.yandex-team.ru/security/xray/pkg/xrayrpc"
	"a.yandex-team.ru/security/yodax/lib/formatters/yodax_out"
)

const (
	Name    = "Yodax"
	Type    = "yodax"
	Version = "0.1"
)

var (
	_ collector.BoxFSCollector = (*Yodax)(nil)
	_ collector.Collector      = (*Yodax)(nil)
)

type (
	Yodax struct {
		command string
	}
)

func New(env collector.Config) *Yodax {
	return &Yodax{
		command: fmt.Sprintf("%s/yodax --silent --format=proto --issues-exit-code=3", env.ContainerDir),
	}
}

func (s *Yodax) Name() string {
	return Name
}

func (s *Yodax) Type() string {
	return Type
}

func (s *Yodax) Version() string {
	return Version
}

func (s *Yodax) Command() string {
	return s.command
}

func (s *Yodax) Sync(_ context.Context) error {
	return nil
}

func (s *Yodax) Requirements() collector.ContainerRequirements {
	return collector.ContainerRequirements{
		Net: false,
	}
}

func (s *Yodax) Deadline() time.Duration {
	return 5 * time.Minute
}

func (s *Yodax) ProcessContainerResult(_ log.Logger, result *collector.ContainerResult, fsID string) (finding *xrayrpc.Finding, err error) {
	exitStatus := result.Status.ExitStatus()
	switch exitStatus {
	case 0:
		// nothing to do
		return
	case 3:
		// ok
	default:
		err = fmt.Errorf("unexpected yodax exit code: %d. stderr: %s", exitStatus, string(result.Stderr))
		return
	}

	var yodaxOut yodax_out.Result
	err = proto.Unmarshal(result.Stdout, &yodaxOut)
	if err != nil {
		err = fmt.Errorf("failed to decode yodax results: %w", err)
		return
	}

	findings := make([]*xrayrpc.YodaxFindingDetail_YodaxInfo, 0, len(yodaxOut.Issues))
	for _, issue := range yodaxOut.Issues {
		findings = append(findings, &xrayrpc.YodaxFindingDetail_YodaxInfo{
			Path:        issue.Path,
			Config:      issue.Config,
			Description: issue.Description,
			Plugin:      issue.Plugin,
			Reason:      issue.Reason,
			Reference:   issue.Reference,
			Summary:     issue.Summary,
			Severity:    parseSeverity(issue.Severity),
		})
	}

	return &xrayrpc.Finding{
		Id: generateFindingID(fsID),
		Details: &xrayrpc.Finding_Yodax{
			Yodax: &xrayrpc.YodaxFindingDetail{
				Problems: findings,
			},
		},
	}, nil
}

func parseSeverity(severity yodax_out.Severity) xrayrpc.YodaxFindingDetail_YodaxSeverity {
	switch severity {
	case yodax_out.Severity_S_HIGH:
		return xrayrpc.YodaxFindingDetail_YS_HIGH
	case yodax_out.Severity_S_MEDIUM:
		return xrayrpc.YodaxFindingDetail_YS_MEDIUM
	case yodax_out.Severity_S_LOW:
		return xrayrpc.YodaxFindingDetail_YS_LOW
	case yodax_out.Severity_S_UNSPECIFIED:
		fallthrough
	default:
		return xrayrpc.YodaxFindingDetail_YS_UNSPECIFIED
	}
}

func generateFindingID(fsID string) string {
	var buf bytes.Buffer
	buf.WriteString(Type)
	buf.WriteByte(':')
	buf.WriteString(fsID)

	return base64.RawURLEncoding.EncodeToString(buf.Bytes())
}
