package xray

import (
	"context"
	"fmt"
	"time"

	grpcMiddleware "github.com/grpc-ecosystem/go-grpc-middleware"
	grpcRetry "github.com/grpc-ecosystem/go-grpc-middleware/retry"
	"google.golang.org/grpc"
	"google.golang.org/grpc/codes"
	"google.golang.org/protobuf/types/known/emptypb"

	"a.yandex-team.ru/infra/yp_service_discovery/golang/resolver"
	"a.yandex-team.ru/infra/yp_service_discovery/golang/wrapper/grpcresolver"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/nop"
	"a.yandex-team.ru/security/xray/pkg/xrayrpc"
)

const (
	DefaultAddr            = "xray.api"
	DefaultRetries         = 3
	DefaultTimeout         = 3 * time.Second
	DefaultBackoffDuration = 100 * time.Millisecond
)

var (
	DefaultClusters = []string{
		resolver.ClusterVLA,
		resolver.ClusterSAS,
	}
)

type Client struct {
	l               log.Structured
	conn            *grpc.ClientConn
	credentials     Credentials
	commonService   xrayrpc.CommonServiceClient
	stageService    xrayrpc.StageServiceClient
	analysisService xrayrpc.AnalysisServiceClient
}

func NewClient(opts ...Option) (client *Client, err error) {
	client = &Client{
		l: &nop.Logger{},
	}

	cfg := &config{
		addr:            grpcresolver.GrpcTarget(DefaultAddr, DefaultClusters...),
		maxRetries:      DefaultRetries,
		backoffDuration: DefaultBackoffDuration,
		timeout:         DefaultTimeout,
	}

	for _, opt := range opts {
		opt(client, cfg)
	}

	grpcRetry.WithBackoff(grpcRetry.BackoffLinear(100 * time.Millisecond))
	client.conn, err = NewConnection(cfg.addr,
		GRPCTransportCredentials(cfg.insecure),
		grpc.WithUnaryInterceptor(
			grpcMiddleware.ChainUnaryClient(
				client.unaryInterceptor,
				grpcRetry.UnaryClientInterceptor(
					grpcRetry.WithMax(uint(cfg.maxRetries)),
					grpcRetry.WithPerRetryTimeout(cfg.timeout),
					grpcRetry.WithCodes(codes.Aborted, codes.Internal, codes.DataLoss),
					grpcRetry.WithBackoff(grpcRetry.BackoffExponentialWithJitter(cfg.backoffDuration, 0.2)),
				),
			),
		),
	)

	if err != nil {
		return nil, fmt.Errorf("failed to create gRPC connection: %w", err)
	}

	client.commonService = xrayrpc.NewCommonServiceClient(client.conn)
	client.stageService = xrayrpc.NewStageServiceClient(client.conn)
	client.analysisService = xrayrpc.NewAnalysisServiceClient(client.conn)
	return client, nil
}

func (c *Client) Close() error {
	if err := c.conn.Close(); err != nil {
		return fmt.Errorf("failed to close gRPC connection: %w", err)
	}
	return nil
}

func (c *Client) Ping(ctx context.Context) error {
	_, err := c.commonService.Ping(ctx, &emptypb.Empty{})
	return err
}

func (c *Client) Schedule(
	ctx context.Context,
	req *xrayrpc.StageScheduleRequest,
) (*xrayrpc.StageScheduleReply, error) {

	return c.stageService.Schedule(ctx, req)
}

func (c *Client) ScheduleLatest(
	ctx context.Context,
	req *xrayrpc.StageScheduleLatestRequest,
) (*xrayrpc.StageScheduleLatestReply, error) {

	return c.stageService.ScheduleLatest(ctx, req)
}

func (c *Client) GetStatus(
	ctx context.Context,
	req *xrayrpc.StageGetStatusRequest,
) (*xrayrpc.StageGetStatusReply, error) {

	return c.stageService.GetStatus(ctx, req)
}

func (c *Client) ListAll(
	ctx context.Context,
	req *xrayrpc.StageListAllRequest,
) (*xrayrpc.StageListAllReply, error) {

	return c.stageService.ListAll(ctx, req)
}

func (c *Client) List(
	ctx context.Context,
	req *xrayrpc.StageListRequest,
) (*xrayrpc.StageListReply, error) {

	return c.stageService.List(ctx, req)
}

func (c *Client) GetResults(
	ctx context.Context,
	stage *xrayrpc.Stage,
) (*xrayrpc.StageGetResultsReply, error) {

	return c.stageService.GetResults(ctx, &xrayrpc.StageGetResultsRequest{
		Stage: stage,
	})
}

func (c *Client) GetAnalysis(
	ctx context.Context,
	analyzeID string,
) (*xrayrpc.AnalysisGetReply, error) {

	return c.analysisService.Get(ctx, &xrayrpc.AnalysisGetRequest{
		Id: analyzeID,
	})
}

func (c *Client) GetAnalysisStatus(
	ctx context.Context,
	analyzeID string,
) (*xrayrpc.AnalysisGetStatusReply, error) {

	return c.analysisService.GetStatus(ctx, &xrayrpc.AnalysisGetStatusRequest{
		Id: analyzeID,
	})
}

func (c *Client) unaryInterceptor(
	ctx context.Context,
	method string,
	req, reply interface{},
	cc *grpc.ClientConn,
	invoker grpc.UnaryInvoker,
	opts ...grpc.CallOption,
) (err error) {

	if requestCredentials := ContextCredentials(ctx); requestCredentials != nil {
		ctx, err = requestCredentials.Set(ctx)
	} else if c.credentials != nil {
		ctx, err = c.credentials.Set(ctx)
	}

	if err != nil {
		return fmt.Errorf("failed to set credentials: %w", err)
	}

	return invoker(ctx, method, req, reply, cc, opts...)
}
