package xray

import (
	"context"
	"fmt"

	"google.golang.org/grpc/metadata"

	"a.yandex-team.ru/library/go/yandex/tvm"
)

type Credentials interface {
	Set(ctx context.Context) (context.Context, error)
}

type TokenCredentials struct {
	Token string
}

func (c *TokenCredentials) Set(ctx context.Context) (context.Context, error) {
	return metadata.AppendToOutgoingContext(ctx, XYaTokenHeader, c.Token), nil
}

type TVMCredentials struct {
	ClientID  tvm.ClientID
	TVMClient tvm.Client
}

func (c *TVMCredentials) Set(ctx context.Context) (context.Context, error) {
	ticket, err := c.TVMClient.GetServiceTicketForID(ctx, c.ClientID)
	if err != nil {
		return ctx, fmt.Errorf("failed to get service ticket for id %d: %w", c.ClientID, err)
	}

	return metadata.AppendToOutgoingContext(ctx, XYaServiceTicketHeader, ticket), nil
}

type UserTicketCredentials struct {
	TVMCredentials
	UserTicket string
}

func (c *UserTicketCredentials) Set(ctx context.Context) (context.Context, error) {
	ctx, err := c.TVMCredentials.Set(ctx)
	if err != nil {
		return ctx, err
	}

	return metadata.AppendToOutgoingContext(ctx, XYaUserTicketHeader, c.UserTicket), nil
}

type credentials struct{}

var credentialsKey credentials

func ContextCredentials(ctx context.Context) Credentials {
	if v := ctx.Value(&credentialsKey); v != nil {
		return v.(Credentials)
	}

	return nil
}

// WithCredentials allows overriding client credentials on per-call basis.
func WithCredentials(ctx context.Context, credentials Credentials) context.Context {
	return context.WithValue(ctx, &credentialsKey, credentials)
}
