package xray

import (
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/yandex/tvm"
)

type config struct {
	addr            string
	insecure        bool
	maxRetries      int
	timeout         time.Duration
	backoffDuration time.Duration
}

type Option func(*Client, *config)

func WithAddr(addr string) Option {
	return func(_ *Client, cfg *config) {
		if addr != "" {
			cfg.addr = addr
		}
	}
}

func WithInsecure(insecure bool) Option {
	return func(_ *Client, cfg *config) {
		cfg.insecure = insecure
	}
}

func WithLogger(logger log.Structured) Option {
	return func(c *Client, _ *config) {
		c.l = logger
	}
}

func WithTokenAuth(oauthToken string) Option {
	return func(c *Client, _ *config) {
		c.credentials = &TokenCredentials{
			Token: oauthToken,
		}
	}
}

func WithTvmAuth(tvmClient tvm.Client, id tvm.ClientID) Option {
	return func(c *Client, _ *config) {
		c.credentials = &TVMCredentials{
			TVMClient: tvmClient,
			ClientID:  id,
		}
	}
}

func WithMaxRetries(retries int) Option {
	return func(_ *Client, cfg *config) {
		cfg.maxRetries = retries
	}
}

func WithBackoffDuration(duration time.Duration) Option {
	return func(_ *Client, cfg *config) {
		cfg.backoffDuration = duration
	}
}

func WithTimeout(timeout time.Duration) Option {
	return func(_ *Client, cfg *config) {
		cfg.timeout = timeout
	}
}
