package main

// Api Auth by OAuth
// requires login scope
// Oauth app: https://oauth.yandex-team.ru/client/00cc76f702994a42ab30bbe4d727dc91
// Get here: https://oauth.yandex-team.ru/authorize?response_type=token&client_id=00cc76f702994a42ab30bbe4d727dc91
// Usage:
//     Authorization: OAuth <...>

// Staff OAuth
// OAuth app to get staff token: https://oauth.yandex-team.ru/client/e4c26c9de39d458c9663b800b1265ccf
// Url to get OAuth token that read staff: https://oauth.yandex-team.ru/authorize?response_type=token&client_id=e4c26c9de39d458c9663b800b1265ccf
//     export YA_STAFF_TOKEN=XXXXXXXXXXXXXXXXX

// ABC OAuth
// OAuth app to get ABC token: https://oauth.yandex-team.ru/client/97f4d3e0016f486c8c6200c19c651687
// Url to get OAuth token that read abc: https://oauth.yandex-team.ru/authorize?response_type=token&client_id=97f4d3e0016f486c8c6200c19c651687
//     export YA_ABC_TOKEN=XXXXXXXXXXXXXXXXX

// Puncher OAuth
// OAuth app to get ABC token: https://oauth.yandex-team.ru/client/a4b9cc023f7244e8b2c7b4fa47c444a6
// Url to get OAuth token that read abc: https://oauth.yandex-team.ru/authorize?response_type=token&client_id=a4b9cc023f7244e8b2c7b4fa47c444a6
//     export PUNCHER_TOKEN=XXXXXXXXXXXXXXXXX

// TVMTOOL
// To use tvmtool compatible with Y.Deploy set env vars DEPLOY_TVM_TOOL_URL and TVMTOOL_LOCAL_AUTHTOKEN
// You can get TVMTOOL_LOCAL_AUTHTOKEN from stdout of tvmtool when running locally.
// Examples:
//     export DEPLOY_TVM_TOOL_URL=http://localhost:9999
//     export TVMTOOL_LOCAL_AUTHTOKEN=...

import (
	"fmt"
	"net/http"
	"os"
	"sort"

	"github.com/labstack/echo/v4"
	echoMiddleware "github.com/labstack/echo/v4/middleware"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/security/ya_resolve/internal/abc"
	"a.yandex-team.ru/security/ya_resolve/internal/bot"
	"a.yandex-team.ru/security/ya_resolve/internal/macros"
	"a.yandex-team.ru/security/ya_resolve/internal/puncher"
	"a.yandex-team.ru/security/ya_resolve/internal/racktables"
	"a.yandex-team.ru/security/ya_resolve/internal/resolver"
	"a.yandex-team.ru/security/ya_resolve/internal/staff"
	"a.yandex-team.ru/security/ya_resolve/internal/walle"
)

type Response struct {
	Status  string      `json:"status"`
	Result  interface{} `json:"result"`
	Message string      `json:"message"`
}

var (
	abcToken     = os.Getenv("YA_ABC_TOKEN")
	staffToken   = os.Getenv("YA_STAFF_TOKEN")
	puncherToken = os.Getenv("PUNCHER_TOKEN")

	logger = func() log.Logger {
		zlog, err := zap.New(zap.ConsoleConfig(log.InfoLevel))
		if err != nil {
			panic(fmt.Sprintf("failed to create logger: %s", err))
		}
		return zlog
	}()

	abcClient = func() *abc.ABC {
		a, err := abc.NewABC(abcToken, abc.WithLogger(logger))
		if err != nil {
			panic(fmt.Sprintf("failed to create abcClient: %s", err))
		}
		return a
	}()

	staffClient = func() *staff.Staff {
		s, err := staff.NewStaff(staffToken, staff.WithLogger(logger))
		if err != nil {
			panic(fmt.Sprintf("failed to create staffClient: %s", err))
		}
		return s
	}()

	racktablesClient = func() *racktables.RacktablesClient {
		s, err := racktables.NewRacktablesClient(racktables.WithLogger(logger))
		if err != nil {
			panic(fmt.Sprintf("failed to create racktablesClient: %s", err))
		}
		return s
	}()

	botClient = func() *bot.BotClient {
		s, err := bot.NewBotClient(bot.WithLogger(logger))
		if err != nil {
			panic(fmt.Sprintf("failed to create botClient: %s", err))
		}
		return s
	}()

	walleClient = func() *walle.Client {
		s, err := walle.NewClient(walle.WithLogger(logger))
		if err != nil {
			panic(fmt.Sprintf("failed to create walleClient: %s", err))
		}
		return s
	}()

	hbfClient = func() *macros.HBFClient {
		s, err := macros.NewHBFClient(macros.WithLogger(logger))
		if err != nil {
			panic(fmt.Sprintf("failed to create hbfClient: %s", err))
		}
		return s
	}()

	puncherClient = func() *puncher.Client {
		c, err := puncher.NewClient(puncherToken, puncher.WithLogger(logger))
		if err != nil {
			panic(fmt.Sprintf("failed to create puncherClient: %v", err))
		}
		return c
	}()

	resolverClient = resolver.NewResolver(
		*abcClient,
		*staffClient,
		*racktablesClient,
		*botClient,
		*walleClient,
		*hbfClient,
		*puncherClient,
		resolver.WithLogger(logger),
	)
)

func resolveStaff(c echo.Context) error {
	staff, err := staff.NewStaff(staffToken)
	if err != nil {
		response := Response{
			Status: "error",
			Result: nil,
		}
		return c.JSON(500, response)
	}
	user := c.Param("user")
	resolutions := staff.Resolve(user)
	response := Response{
		Status: "ok",
		Result: resolutions,
	}
	return c.JSON(http.StatusOK, response)
}

func resolveABCServiceBySlug(c echo.Context) error {
	abc, err := abc.NewABC(abcToken)
	if err != nil {
		response := Response{
			Status: "error",
			Result: nil,
		}
		return c.JSON(500, response)
	}

	service := c.Param("service")

	resolutions, err := abc.ResolveSlug(service)
	if err == nil {
		response := Response{
			Status: "ok",
			Result: resolutions,
		}
		return c.JSON(http.StatusOK, response)
	} else {
		response := Response{
			Status: "error",
			Result: nil,
		}
		return c.JSON(500, response)
	}
}

func racktablesOwnersResolver(c echo.Context) error {
	owners, err := racktablesClient.GetOwners()
	if err != nil {
		response := Response{
			Status: "error",
			Result: nil,
		}
		return c.JSON(500, response)
	} else {
		response := Response{
			Status: "ok",
			Result: owners.GetAllVSOwners("trust-xmlrpc.yandex.net"),
		}
		return c.JSON(http.StatusOK, response)
	}
}

func getABCServiceMembers(c echo.Context) error {
	abc, err := abc.NewABC(abcToken)
	if err != nil {
		response := Response{
			Status: "error",
			Result: nil,
		}
		return c.JSON(500, response)
	}

	service := c.Param("service")
	members, err := abc.GetServiceMembers(service)
	if err != nil {
		response := Response{
			Status: "error",
			Result: nil,
		}
		return c.JSON(500, response)
	} else {
		response := Response{
			Status: "ok",
			Result: members,
		}
		return c.JSON(http.StatusOK, response)
	}
}

func mainResolver(c echo.Context) error {

	inputValue := c.QueryParam("value")
	if len(inputValue) == 0 {
		response := Response{
			Status:  "error",
			Message: "no query param 'value' specified",
		}
		return c.JSON(500, response)
	}

	// parse input_type
	inputType := resolver.ValueType(c.QueryParam("type"))
	if len(inputType) == 0 {
		inputType = resolver.TypeUnknown
	}

	// resolve unknown or specified input value type
	resolutions := resolverClient.ResolveType(inputValue, inputType)

	// filter only success resolutionss
	successOnly := c.QueryParam("success_only") == "1"
	if successOnly {
		var successResolutions []resolver.Resolution
		for _, resolution := range resolutions {
			if resolution.Status == resolver.ResStatusSuccess {
				successResolutions = append(successResolutions, resolution)
			}
		}
		resolutions = successResolutions
	}

	// Estimate Weights
	skipWeightsEstimation := c.QueryParam("no_weights") == "1"
	if !skipWeightsEstimation {
		for i := range resolutions {
			resolutions[i].Weight = resolver.EstimateWeights(resolutions[i])
		}
	}

	// Sort by weight and flow depth
	skipSorting := c.QueryParam("dont_sort") == "1"
	if !skipSorting {
		sort.Slice(resolutions, func(i, j int) bool {
			if resolutions[i].Weight != resolutions[j].Weight {
				return resolutions[i].Weight > resolutions[j].Weight
			}
			return len(resolutions[i].Flow) < len(resolutions[j].Flow)
		})
	}

	response := Response{
		Status: "ok",
		Result: resolutions,
	}

	return c.JSON(http.StatusOK, response)
}

func main() {
	logger.Fmt().Infof("initializing app")
	maxprocs.AdjustYP()

	// allowedLogins := []string{
	// 	"a-abakumov",
	// 	"procenkoeg",
	// 	"ezaitov",
	// 	"buglloc",
	// 	"horus",
	// 	"gots",
	// 	"shikari",
	// 	"shashial",
	// 	"limetime",
	// 	"spellanser",
	// 	"anton-k",
	// 	"avkov",
	// 	"ybuchnev",
	// 	"kaleda",
	// 	"melkikh",
	// }

	// acl := make(map[string]bool)
	// for _, login := range allowedLogins {
	// 	acl[login] = true
	// }

	e := echo.New()
	e.Debug = true

	e.Use(echoMiddleware.RecoverWithConfig(echoMiddleware.RecoverConfig{
		StackSize: 1 << 10, // 1 KB
	}))

	e.GET("/ping", func(c echo.Context) error {
		return c.String(http.StatusOK, "pong")
	})

	apiV1 := e.Group("/api/v1")
	// apiV1.Use(middleware.AuthMiddleware(acl))

	apiV1.GET("/staff/resolve/:user", resolveStaff)
	apiV1.GET("/abc/members/slug/:service", getABCServiceMembers)
	apiV1.GET("/abc/resolve/slug/:service", resolveABCServiceBySlug)
	apiV1.GET("/racktables/owners", racktablesOwnersResolver)
	apiV1.GET("/resolve", mainResolver)

	port := os.Getenv("ECHO_PORT")
	if port == "" {
		port = "3333"
	}

	e.Logger.Fatal(e.Start(":" + port))
}
