package abc

import (
	"crypto/tls"
	"fmt"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/certifi"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/nop"
)

type (
	RoleName  string
	ResStatus string
)

const (
	BaseABCAPIURL = "https://abc-back.yandex-team.ru/api/v4"

	RoleServiceManagement       RoleName = "RoleServiceManagement"       // role.code = ANY, role.scope.slug = "services_management"
	RoleAdministration          RoleName = "RoleAdministration"          // role.code = ANY, role.scope.slug = "administration"
	RoleAny                     RoleName = "RoleAny"                     // role.code = ANY, role.scope.slug = ANY
	RoleAnyNotServiceManagement RoleName = "RoleAnyNotServiceManagement" // role.code = ANY, role.scope.slug != "services_management"
	RoleDuty                    RoleName = "RoleDuty"                    // role.scope.slug = "dutywork", role.code = ANY
	RoleDutyCurrently           RoleName = "RoleDutyCurrently"           // role.scope.slug = "dutywork", role.code = "duty"
	RoleDutyResponsible         RoleName = "RoleDutyResponsible"         // role.scope.slug = "dutywork", role.code = "responsible_for_duty"
	// RoleHead RoleName = "RoleManagementHead"                    // role.scope.slug = "services_management", role.code = "product_head"
	// RoleDeputyHead RoleName = "RoleManagementDeputyHead"        // role.scope.slug = "services_management", role.code = "product_deputy_head"
	// RoleAnalitics RoleName = "RoleAnalitics"                    // role.code = ANY, role.scope.slug = "analitics"
	// RoleDBManagement RoleName = "RoleDBManagement"              // role.code = ANY, role.scope.slug = "db_management"
	// RoleQuotasManagement RoleName = "RoleQuotasManagement"      // role.code = ANY, role.scope.slug = "quotas_management"
	// RoleRobotsManagement RoleName = "RoleRobotsManagement"      // role.code = ANY, role.scope.slug = "robots_management"
	// RoleHardwareManagement RoleName = "RoleHardwareManagement"  // role.code = ANY, role.scope.slug = "hardware_management"
	// RoleDevelopment RoleName = "RoleDevelopment"                // role.code = ANY, role.scope.slug = "development"
	// RoleTVMManagement RoleName = "RoleTVMManagement"            // role.code = ANY, role.scope.slug = "tvm_management"

	ResolutionFail       ResStatus = "ResolutionFail"
	ResolutionProcessing ResStatus = "ResolutionProcessing"
	ResolutionSuccess    ResStatus = "ResolutionSuccess"
)

var (
	DefaultRoleSequence = []RoleName{RoleDutyCurrently, RoleDuty, RoleAdministration, RoleAnyNotServiceManagement, RoleAny}

	Suppliers = map[string]int{
		"bot": 385,
	}

	ResourceTypes = map[string]int{
		"tvm_app": 47,
	}
)

type ABCResolution struct {
	ResolutionStatus ResStatus `json:"resolution_status"`
	ResolveDepth     int       `json:"resolve_depth"`
	ResolveFlow      string    `json:"resolve_flow"`
	User             string    `json:"user"`
}

type ABC struct {
	Token              string
	ResolutionSequence []RoleName
	Client             resty.Client
	Logger             log.Logger
}

func NewHTTPClient(token string) (*resty.Client, error) {
	certPool, err := certifi.NewCertPool()
	if err != nil {
		return nil, err
	}

	httpClient := resty.New().
		SetRetryCount(2).
		SetHeader("Content-Type", "application/json").
		SetHeader("Authorization", "OAuth "+token).
		SetHeader("User-Agent", "ya_resolve <security@yandex-team.ru>").
		SetTLSClientConfig(&tls.Config{RootCAs: certPool})

	return httpClient, nil
}

func NewABC(token string, options ...Option) (*ABC, error) {
	httpClient, err := NewHTTPClient(token)
	if err != nil {
		return nil, err
	}

	abc := ABC{
		Token:              token,
		ResolutionSequence: DefaultRoleSequence,
		Client:             *httpClient,
		Logger:             &nop.Logger{},
	}

	for _, opt := range options {
		opt(&abc)
	}

	return &abc, nil
}

func getABCResolutions(allMembers []ABCServiceMembers, role RoleName, serviceSlug string) []ABCResolution {
	var resolutions []ABCResolution

	managementLogins := make(map[string]bool)

	if role == "RoleAnyNotServiceManagement" {
		for _, member := range allMembers {
			if member.Role.Scope.Slug == "services_management" {
				managementLogins[member.Person.Login] = true
			}
		}
	}

	for _, member := range allMembers {

		switch role {
		case RoleDuty:
			if member.Role.Scope.Slug == "dutywork" {
				resolution := ABCResolution{
					ResolutionStatus: ResolutionSuccess,
					ResolveDepth:     1,
					ResolveFlow:      fmt.Sprintf("Duty on service (%s)", serviceSlug),
					User:             member.Person.Login,
				}
				resolutions = append(resolutions, resolution)
			}

		case RoleDutyCurrently:
			if member.Role.Scope.Slug == "dutywork" && member.Role.Code == "duty" {
				resolution := ABCResolution{
					ResolutionStatus: ResolutionSuccess,
					ResolveDepth:     1,
					ResolveFlow:      fmt.Sprintf("Current duty on service (%s)", serviceSlug),
					User:             member.Person.Login,
				}
				resolutions = append(resolutions, resolution)
			}

		case RoleAdministration:
			if member.Role.Scope.Slug == "administration" {
				resolution := ABCResolution{
					ResolutionStatus: ResolutionSuccess,
					ResolveDepth:     1,
					ResolveFlow:      fmt.Sprintf("Administration role in service (%s)", serviceSlug),
					User:             member.Person.Login,
				}
				resolutions = append(resolutions, resolution)
			}

		case RoleAnyNotServiceManagement:
			if member.Role.Scope.Slug != "services_management" {
				if !managementLogins[member.Person.Login] {
					resolution := ABCResolution{
						ResolutionStatus: ResolutionSuccess,
						ResolveDepth:     1,
						ResolveFlow:      fmt.Sprintf("Not service management role in service (%s)", serviceSlug),
						User:             member.Person.Login,
					}
					resolutions = append(resolutions, resolution)
				}
			}

		case RoleAny:
			resolution := ABCResolution{
				ResolutionStatus: ResolutionSuccess,
				ResolveDepth:     1,
				ResolveFlow:      fmt.Sprintf("Member in service (%s)", serviceSlug),
				User:             member.Person.Login,
			}
			resolutions = append(resolutions, resolution)

		}

	}

	return resolutions
}

func (abc ABC) ResolveSlug(serviceSlug string) ([]ABCResolution, error) {

	serviceMembers, err := abc.GetServiceMembers(serviceSlug)
	if err != nil {
		return nil, err
	}

	for _, role := range abc.ResolutionSequence {
		resolutions := getABCResolutions(serviceMembers, role, serviceSlug)
		if len(resolutions) > 0 {
			return resolutions, nil
		}
	}

	return nil, fmt.Errorf("no matching service members for role sequence that was used")
}

func (abc ABC) GetServiceMembers(serviceSlug string) ([]ABCServiceMembers, error) {

	var response ABCResponseMembers

	resp, err := abc.Client.R().
		SetQueryParams(map[string]string{
			"service__slug": serviceSlug,
		}).
		SetResult(&response).
		Get(BaseABCAPIURL + "/services/members/")

	if err != nil {
		return nil, err
	}

	if !resp.IsSuccess() {
		return nil, fmt.Errorf("ABC::GetServiceMembers. resp.StatusCode != 200. Status code: %d", resp.StatusCode())
	}
	return response.Results, nil
}

func (abc ABC) ResolveServiceIDToMembers(serviceID int) ([]ABCResolution, error) {

	serviceMembers, err := abc.GetServiceMembersByServiceID(serviceID)
	if err != nil {
		return nil, err
	}

	for _, role := range abc.ResolutionSequence {
		resolutions := getABCResolutions(serviceMembers, role, fmt.Sprintf("%d", serviceID))
		if len(resolutions) > 0 {
			return resolutions, nil
		}
	}

	return nil, fmt.Errorf("no matching service members for role sequence that was used")
}

func (abc ABC) GetServiceMembersByServiceID(serviceID int) ([]ABCServiceMembers, error) {

	var response ABCResponseMembers

	resp, err := abc.Client.R().
		SetQueryParams(map[string]string{
			"service": fmt.Sprintf("%d", serviceID),
		}).
		SetResult(&response).
		Get(BaseABCAPIURL + "/services/members/")

	if err != nil {
		return nil, err
	}

	if !resp.IsSuccess() {
		return nil, fmt.Errorf("ABC::GetServiceMembersByServiceID. resp.StatusCode != 200. Status code: %d", resp.StatusCode())
	}
	return response.Results, nil
}

// Returns list of consumers service slug
func (abc ABC) GetConsumerSlugsByResourceExternalIDAndSupplierID(resourceExternalID string, supplierID int) ([]string, error) {
	var response ABCConsumerResponse

	resp, err := abc.Client.R().
		SetQueryParams(map[string]string{
			"resource__external_id": resourceExternalID,
			"supplier":              fmt.Sprintf("%d", supplierID),
			"state__in":             "requested,approved,granting,granted",
		}).
		SetResult(&response).
		Get(BaseABCAPIURL + "/resources/consumers/")

	if err != nil {
		return nil, err
	}

	if !resp.IsSuccess() {
		return nil, fmt.Errorf("ABC::GetConsumerByResourceExternalID. !resp.IsSuccess(). Status code: %d", resp.StatusCode())
	}

	slugDict := make(map[string]bool)
	for _, res := range response.Results {
		slugDict[res.Service.Slug] = true
	}

	var consumerSlugs []string
	for slug := range slugDict {
		consumerSlugs = append(consumerSlugs, slug)
	}

	if len(consumerSlugs) == 0 {
		return nil, fmt.Errorf("no consumers matched to external resource id %s", resourceExternalID)
	}

	return consumerSlugs, nil
}

func (abc ABC) GetSupplierIDBySlug(slug string) (*int, error) {
	val, ok := Suppliers[slug]
	if !ok {
		return nil, fmt.Errorf("supplier (%s) not found", slug)
	}
	return &val, nil
}

// Returns list of consumers service slug
func (abc ABC) GetConsumerSlugsByResourceExternalIDAndResourceType(resourceExternalID string, resourceTypeID int) ([]string, error) {
	var response ABCConsumerResponse

	resp, err := abc.Client.R().
		SetQueryParams(map[string]string{
			"resource__external_id": resourceExternalID,
			"type":                  fmt.Sprintf("%d", resourceTypeID),
		}).
		SetResult(&response).
		Get(BaseABCAPIURL + "/resources/consumers/")

	if err != nil {
		return nil, err
	}

	if !resp.IsSuccess() {
		return nil, fmt.Errorf("ABC::GetConsumerByResourceExternalID. !resp.IsSuccess(). Status code: %d", resp.StatusCode())
	}

	slugDict := make(map[string]bool)
	for _, res := range response.Results {
		slugDict[res.Service.Slug] = true
	}

	var consumerSlugs []string
	for slug := range slugDict {
		consumerSlugs = append(consumerSlugs, slug)
	}

	if len(consumerSlugs) == 0 {
		return nil, fmt.Errorf("no consumers matched to external resource id %s", resourceExternalID)
	}

	return consumerSlugs, nil
}

func (abc ABC) GetResourceTypeID(resourceName string) (*int, error) {
	val, ok := ResourceTypes[resourceName]
	if !ok {
		return nil, fmt.Errorf("resource with name (%s) not found", resourceName)
	}
	return &val, nil
}
