package bot

import (
	"crypto/tls"
	"fmt"
	"strings"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/certifi"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/nop"
)

type BotClient struct {
	Client resty.Client
	Logger log.Logger
}

type BotResponseOSInfo struct {
	InstanceNumber string `json:"instance_number"` // 100512500
	StatusName     string `json:"status_name"`     // OPERATION
	ItemSegment1   string `json:"item_segment1"`   // GB/G250-G52/MG50-G21/8T2.5/2U/2P
	ItemSegment2   string `json:"item_segment2"`   // XEONE5-2660V4/TESLAM40
	ItemSegment3   string `json:"item_segment3"`   // SERVERS
}

type BotResponse struct {
	Res int                 `json:"res"`
	OS  []BotResponseOSInfo `json:"os"`
	Msg *string             `json:"msg"`
}

func (r BotResponse) IsSuccess() bool {
	return r.Res == 1
}

func NewHTTPClient() (*resty.Client, error) {
	certPool, err := certifi.NewCertPool()
	if err != nil {
		return nil, err
	}

	httpClient := resty.New().
		SetRetryCount(2).
		SetHeader("User-Agent", "ya_resolve <security@yandex-team.ru>").
		SetTLSClientConfig(&tls.Config{RootCAs: certPool})

	return httpClient, nil
}

func NewBotClient(options ...Option) (*BotClient, error) {
	httpClient, err := NewHTTPClient()
	if err != nil {
		return nil, err
	}

	botClient := BotClient{
		Client: *httpClient,
		Logger: &nop.Logger{},
	}

	for _, opt := range options {
		opt(&botClient)
	}

	return &botClient, nil
}

func (b *BotClient) GetBotInstanceNumbersByMACAddress(mac string) ([]string, error) {

	var result BotResponse

	resp, err := b.Client.R().
		SetQueryParams(map[string]string{
			"mac":    strings.Replace(mac, ":", "", -1),
			"format": "json",
		}).
		SetResult(&result).
		Get("https://bot.yandex-team.ru/api/osinfo.php")

	if err != nil {
		return nil, err
	}

	if !resp.IsSuccess() {
		return nil, fmt.Errorf("BotClient::GetBotInstanceNumbersByMACAddress. !resp.IsSuccess(). Status code: %d", resp.StatusCode())
	}

	if !result.IsSuccess() {
		return nil, fmt.Errorf("BotClient::GetBotInstanceNumbersByMACAddress. !result.IsSuccess(). %s", *result.Msg)
	}

	instanceNumbers := make([]string, len(result.OS))
	for i, os := range result.OS {
		instanceNumbers[i] = os.InstanceNumber
	}

	return instanceNumbers, nil
}
