package macros

import (
	"sync"

	"github.com/jasonlvhit/gocron"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/nop"
)

type MacrosCache struct {
	cron       *gocron.Scheduler
	hbfClient  HBFClient
	logger     log.Logger
	mu         *sync.Mutex
	mtnMapping map[ProjectID][]Target
}

type CacheOption func(c *MacrosCache)

func CacheWithLogger(logger log.Logger) CacheOption {
	return func(c *MacrosCache) {
		c.logger = logger
	}
}

func NewCache(hbfClient HBFClient, options ...CacheOption) MacrosCache {
	cache := MacrosCache{
		cron:       gocron.NewScheduler(),
		hbfClient:  hbfClient,
		logger:     &nop.Logger{},
		mu:         &sync.Mutex{},
		mtnMapping: make(map[ProjectID][]Target),
	}

	for _, op := range options {
		op(&cache)
	}

	return cache
}

func (c *MacrosCache) UpdateCache() {
	c.logger.Fmt().Infof("MacrosCache. updateCache started...")
	defer c.logger.Fmt().Infof("MacrosCache. updateCache finished...")

	mtnMapping, err := c.hbfClient.FetchMTNMapping()
	if err != nil {
		c.logger.Fmt().Infof("error on macros fetching: %v", err)
		return
	}

	c.mu.Lock()
	c.mtnMapping = mtnMapping
	c.mu.Unlock()
}

func (c *MacrosCache) StartCron() {
	c.logger.Fmt().Infof("cron job started...")
	c.cron.Every(3).Hours().Do(c.UpdateCache)
	go func() {
		<-c.cron.Start()
	}()
}

func (c *MacrosCache) GetMacrosByProjectID(projectID ProjectID) ([]string, bool) {
	targets, exists := c.mtnMapping[projectID]
	if !exists {
		return nil, false
	}

	macros := make([]string, len(targets))
	for i, target := range targets {
		macros[i] = target.Macro
	}

	return macros, true
}
