package macros

import (
	"crypto/tls"
	"fmt"
	"net"
	"strconv"
	"strings"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/certifi"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/nop"
)

type Target struct {
	Macro   string
	Network net.IPNet
}

type ProjectID uint32

type HBFClient struct {
	httpc  resty.Client
	Logger log.Logger
}

type Option func(c *HBFClient)

func WithLogger(logger log.Logger) Option {
	return func(c *HBFClient) {
		c.Logger = logger
		c.httpc.SetLogger(logger.Fmt())
	}
}

func NewHBFClient(options ...Option) (*HBFClient, error) {
	certPool, err := certifi.NewCertPool()
	if err != nil {
		return nil, err
	}

	httpClient := resty.New().
		SetRetryCount(2).
		SetHeader("User-Agent", "ya_resolve <security@yandex-team.ru>").
		SetTLSClientConfig(&tls.Config{RootCAs: certPool})

	hbfClient := HBFClient{
		httpc:  *httpClient,
		Logger: &nop.Logger{},
	}

	for _, opt := range options {
		opt(&hbfClient)
	}

	return &hbfClient, nil
}

func (c HBFClient) FetchMTNMapping() (map[ProjectID][]Target, error) {
	results := make(map[string][]string)

	resp, err := c.httpc.R().
		SetQueryParams(map[string]string{
			"net_only": "1",
		}).
		SetResult(&results).
		Get("https://hbf.yandex.net/dump-all-macros")

	if err != nil {
		c.Logger.Fmt().Infof("error on fetching macros from hbf: %s", err)
		return nil, err
	}

	if !resp.IsSuccess() {
		c.Logger.Fmt().Infof("error on fetching macros from hbf. status code: %d", resp.StatusCode())
		return nil, fmt.Errorf("error on fetching macros from hbf. status code: %d", resp.StatusCode())
	}

	MTNMapping := make(map[ProjectID][]Target)
	for macro, networks := range results {
		for _, network := range networks {
			parts := strings.Split(network, "@")
			if len(parts) != 2 {
				continue
			}

			pid := parts[0]
			// TODO: Support project_id with mask. Like "fa00/23", "10000/16", etc...
			// projectIDParts := strings.Split(pid, "/")
			// if len(projectIDParts) == 2 {
			// 	pid = projectIDParts[0]
			// }

			u, err := strconv.ParseUint(pid, 16, 32)
			if err != nil {
				c.Logger.Fmt().Infof("error on project_id as uint32 parsing: %s", err)
				// fmt.Printf("error on project_id as uint32 parsing: %s\n", err)
				continue
			}
			projectID := ProjectID(u)

			_, netPtr, err := net.ParseCIDR(parts[1])
			if err != nil {
				c.Logger.Fmt().Infof("error on parsing network %s: %s", parts[1], err)
				// fmt.Printf("error on parsing network %s: %s\n", parts[1], err)
				continue
			}

			if _, exists := MTNMapping[projectID]; !exists {
				MTNMapping[projectID] = append(MTNMapping[projectID], Target{
					Macro:   macro,
					Network: *netPtr,
				})
				continue
			}

			newNetMaskSize, _ := netPtr.Mask.Size()
			existingNetMaskSize, _ := MTNMapping[projectID][0].Network.Mask.Size()

			// get the list of targets with the biggest mask size
			switch {
			case newNetMaskSize == existingNetMaskSize:
				MTNMapping[projectID] = append(MTNMapping[projectID], Target{
					Macro:   macro,
					Network: *netPtr,
				})
			case newNetMaskSize > existingNetMaskSize:
				MTNMapping[projectID] = []Target{
					Target{
						Macro:   macro,
						Network: *netPtr,
					},
				}
			}
		}
	}

	return MTNMapping, nil
}
