package middleware

import (
	"log"
	"strings"

	"github.com/labstack/echo/v4"

	"a.yandex-team.ru/library/go/yandex/blackbox"
	"a.yandex-team.ru/library/go/yandex/blackbox/httpbb"
	"a.yandex-team.ru/library/go/yandex/tvm/tvmtool"
)

func getUserByOauth(bb blackbox.Client, c echo.Context) *blackbox.User {
	authHeader := c.Request().Header.Get("Authorization")

	if authHeader == "" {
		log.Printf("No Authorization header")
		return nil
	}

	if !strings.HasPrefix(authHeader, "OAuth ") {
		log.Printf("No Oauth prefix")
		return nil
	}

	oauth := strings.TrimPrefix(authHeader, "OAuth ")

	response, err := bb.OAuth(
		c.Request().Context(),
		blackbox.OAuthRequest{
			OAuthToken: oauth,
			UserIP:     c.RealIP(),
			Scopes:     []string{"login:info"},
		},
	)

	if err != nil {
		log.Printf("bb returned error on oath: %v\n", err)
		if !blackbox.IsUnauthorized(err) {
			log.Printf("failed to check authorized oauth: %s\n", err)
		}
		return nil
	}

	return &response.User
}

func getUserBySessionID(bb blackbox.Client, c echo.Context) *blackbox.User {
	sessid, err := c.Cookie("Session_id")
	if err != nil {
		return nil
	}

	rsp, err := bb.SessionID(
		c.Request().Context(),
		blackbox.SessionIDRequest{
			SessionID: sessid.Value,
			UserIP:    c.RealIP(),
			Host:      c.Request().Host,
		},
	)

	if err != nil {
		log.Printf("failed to check user session: %s\n", err)
		return nil
	}

	return &rsp.User
}

func AuthMiddleware(acl map[string]bool) echo.MiddlewareFunc {
	tvmClient, err := tvmtool.NewDeployClient()
	if err != nil {
		// TODO: return err
		panic(err)
	}

	bb, err := httpbb.NewIntranet(httpbb.WithTVM(tvmClient))
	if err != nil {
		panic(err)
	}

	return func(next echo.HandlerFunc) echo.HandlerFunc {

		return func(c echo.Context) error {

			// try oauth
			user := getUserByOauth(bb, c)
			if user == nil {
				// try sessionid
				user = getUserBySessionID(bb, c)
				if user == nil {
					return c.JSON(403, "Go away")
				}
			}

			if !acl[user.Login] {
				return c.JSON(403, "Go away "+user.Login)
			}

			log.Printf("Succes auth" + user.Login)

			return next(c)
		}
	}
}
