package puncher

import (
	"crypto/tls"
	"fmt"
	"strings"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/certifi"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/nop"
)

const StatusError = "error"
const StatusSuccess = "success"

type ResponsiblesResponse struct {
	Responsibles []string `json:"responsibles"`
	Status       string   `json:"status"`
	Message      *string  `json:"message"`
}

type Client struct {
	Client resty.Client
	Logger log.Logger
	Token  string
}

func NewHTTPClient(token string) (*resty.Client, error) {
	certPool, err := certifi.NewCertPool()
	if err != nil {
		return nil, err
	}

	httpClient := resty.New().
		SetRetryCount(2).
		SetHeader("Authorization", "OAuth "+token).
		SetHeader("User-Agent", "ya_resolve <security@yandex-team.ru>").
		SetTLSClientConfig(&tls.Config{RootCAs: certPool})

	return httpClient, nil
}

func NewClient(token string, options ...Option) (*Client, error) {
	httpClient, err := NewHTTPClient(token)
	if err != nil {
		return nil, err
	}

	client := Client{
		Client: *httpClient,
		Logger: &nop.Logger{},
		Token:  token,
	}

	for _, opt := range options {
		opt(&client)
	}

	return &client, nil
}

func (c Client) GetResponsibles(IPOrNetworkOrMacro string) ([]string, error) {

	var response ResponsiblesResponse

	resp, err := c.Client.R().
		SetQueryParams(map[string]string{
			"q": IPOrNetworkOrMacro,
		}).
		SetResult(&response).
		Get("https://api.puncher.yandex-team.ru/api/dynfw/responsibles")

	if err != nil {
		return nil, err
	}

	if !resp.IsSuccess() {
		return nil, fmt.Errorf("Puncher::GetResponsibles. resp.StatusCode != 200. Status code: %d", resp.StatusCode())
	}

	if response.Status != StatusSuccess {
		if response.Message != nil {
			return nil, fmt.Errorf("Puncher::GetResponsibles. response.Status = %s. Message: %s", response.Status, *response.Message)
		} else {
			return nil, fmt.Errorf("Puncher::GetResponsibles. response.Status = %s. No message", response.Status)
		}
	}

	var responsibles []string

	for _, value := range response.Responsibles {
		person := strings.Trim(value, "%")
		responsibles = append(responsibles, person)
	}
	return responsibles, nil
}
