package netsbyprojectcache

import (
	"context"
	"sync"
	"time"

	"a.yandex-team.ru/security/ya_resolve/internal/racktables"
)

type NetsByProjectCache struct {
	RacktablesClient       racktables.RacktablesClient
	PeriodicSyncingContext context.Context
	ContextCancelFunc      context.CancelFunc
	LastSyncTimestamp      int64
	PeriodSyncingIsOn      bool
	MapNetsByProject       map[string]racktables.RacktablesNetsByProject
	mu                     *sync.Mutex
}

func New(racktablesClient racktables.RacktablesClient) NetsByProjectCache {
	return NetsByProjectCache{
		RacktablesClient: racktablesClient,
		mu:               &sync.Mutex{},
	}
}

func (poc *NetsByProjectCache) Sync() {
	poc.updateLastSyncTimestamp()
	poc.RacktablesClient.Logger.Infof("NetsByProjectCache:Sync. Started")
	defer poc.RacktablesClient.Logger.Infof("NetsByProjectCache:Sync. Finished")

	mapNetsByProject, err := poc.RacktablesClient.FetchNetsByProject()
	if err != nil {
		poc.RacktablesClient.Logger.Warnf("NetsByProjectCache:Sync. Error syncing nets_by_project: %v.\n", err)
	} else {
		poc.mu.Lock()
		poc.MapNetsByProject = mapNetsByProject
		poc.mu.Unlock()
	}
}

func (poc *NetsByProjectCache) updateLastSyncTimestamp() {
	poc.LastSyncTimestamp = time.Now().Unix()
}

func (poc NetsByProjectCache) GetInfoByNetwork(network string) (racktables.RacktablesNetsByProject, bool) {
	poc.mu.Lock()
	info, ok := poc.MapNetsByProject[network]
	infoCopy := info
	poc.mu.Unlock()

	if !ok {
		return infoCopy, false
	}

	return infoCopy, true
}

// func (poc NetsByProjectCache) GetMacroOwnerSubjects(macro string) []racktables.SubjectInfo {
// 	return poc.NetsByProjectCache.GetAllMacroOwners(macro)
// }

func (poc *NetsByProjectCache) StartPeriodicSyncing(periodSeconds int64) bool {
	if poc.PeriodSyncingIsOn {
		return false
	}

	poc.PeriodSyncingIsOn = true

	syncer := func(ctx context.Context) {
		for {
			select {
			case <-ctx.Done():
				return
			default:
				if time.Now().Unix()-poc.LastSyncTimestamp > periodSeconds {
					poc.Sync()
				} else {
					time.Sleep(1 * time.Second)
				}
			}
		}
	}
	poc.PeriodicSyncingContext, poc.ContextCancelFunc = context.WithCancel(context.Background())

	go syncer(poc.PeriodicSyncingContext)

	return true
}

func (poc *NetsByProjectCache) StopPeriodicSyncing() bool {
	if !poc.PeriodSyncingIsOn {
		return false
	}

	poc.ContextCancelFunc()
	poc.PeriodSyncingIsOn = false
	return true
}
