package racktables

import (
	"fmt"
)

type SubjType string

const (
	SubjTypeService     SubjType = "service"
	SubjTypeServiceRole SubjType = "servicerole"
	SubjTypeDepartment  SubjType = "department"
	SubjTypeUser        SubjType = "user"
)

type SubjectInfo struct {
	Subj     string   `json:"subj"`
	SubjType SubjType `json:"subj_type"`
}

type OwnersEntry struct {
	FromVlan []SubjectInfo `json:"fromvlan"`
	FW       []SubjectInfo `json:"fw"`
	FWUse    []SubjectInfo `json:"fw_use"`
	ToVlan   []SubjectInfo `json:"tovlan"`
}

type RacktablesOwners struct {
	VS   map[string]OwnersEntry `json:"vs"`
	VLAN map[string]OwnersEntry `json:"vlan"`
	FWS  map[string]OwnersEntry `json:"fws"`
}

func extractSubjectInfosFromOwnersEntries(ownersEntry OwnersEntry) []SubjectInfo {
	allSubjectInfosDict := make(map[SubjectInfo]bool)
	for _, subjInfo := range ownersEntry.FromVlan {
		allSubjectInfosDict[subjInfo] = true
	}
	for _, subjInfo := range ownersEntry.FW {
		allSubjectInfosDict[subjInfo] = true
	}
	for _, subjInfo := range ownersEntry.FWUse {
		allSubjectInfosDict[subjInfo] = true
	}
	for _, subjInfo := range ownersEntry.ToVlan {
		allSubjectInfosDict[subjInfo] = true
	}

	allSubjectInfos := make([]SubjectInfo, len(allSubjectInfosDict))
	i := 0
	for subjInfo := range allSubjectInfosDict {
		allSubjectInfos[i] = subjInfo
		i++
	}

	return allSubjectInfos
}

func (po *RacktablesOwners) GetAllVSOwners(vs string) []SubjectInfo {
	ownersEntry, ok := po.VS[vs]
	if !ok {
		return nil
	}
	return extractSubjectInfosFromOwnersEntries(ownersEntry)
}

func (po *RacktablesOwners) GetAllFQDNOwners(fqdn string) []SubjectInfo {
	return po.GetAllVSOwners(fqdn)
}

func (po *RacktablesOwners) GetAllVLANOwners(vlan string) []SubjectInfo {
	ownersEntry, ok := po.VLAN[vlan]
	if !ok {
		return nil
	}
	return extractSubjectInfosFromOwnersEntries(ownersEntry)
}

func (po *RacktablesOwners) GetAllFWSOwners(fws string) []SubjectInfo {
	ownersEntry, ok := po.FWS[fws]
	if !ok {
		return nil
	}
	return extractSubjectInfosFromOwnersEntries(ownersEntry)
}

func (po *RacktablesOwners) GetAllMacroOwners(macro string) []SubjectInfo {
	return po.GetAllFWSOwners(macro)
}

func (p *RacktablesClient) GetOwners() (*RacktablesOwners, error) {

	var result RacktablesOwners

	resp, err := p.Client.R().
		SetResult(&result).
		Get("https://racktables.yandex.net/export/owners.json")

	if err != nil {
		return nil, err
	}

	if !resp.IsSuccess() {
		return nil, fmt.Errorf("RacktablesClient::GetOwners. !resp.IsSuccess(). Status code: %d", resp.StatusCode())
	}

	return &result, nil
}
