package ownerscache

import (
	"context"
	"time"

	"a.yandex-team.ru/security/ya_resolve/internal/racktables"
)

type RacktablesOwnersCache struct {
	RacktablesClient       racktables.RacktablesClient
	RacktablesOwners       racktables.RacktablesOwners
	PeriodicSyncingContext context.Context
	ContextCancelFunc      context.CancelFunc
	LastSyncTimestamp      int64
	PeriodSyncingIsOn      bool
}

func NewRacktablesOwnersCache(RacktablesClient racktables.RacktablesClient) RacktablesOwnersCache {
	return RacktablesOwnersCache{
		RacktablesClient: RacktablesClient,
	}
}

func (poc *RacktablesOwnersCache) Sync() {
	poc.updateLastSyncTimestamp()
	poc.RacktablesClient.Logger.Infof("RacktablesOwnersCache:Sync. Started")
	defer poc.RacktablesClient.Logger.Infof("RacktablesOwnersCache:Sync. Finished")

	owners, err := poc.RacktablesClient.GetOwners()
	if err != nil {
		poc.RacktablesClient.Logger.Warnf("RacktablesOwnersCache:Sync. Error syncing owners: %v.\n", err)
	} else {
		poc.RacktablesOwners = *owners
	}
}

func (poc *RacktablesOwnersCache) updateLastSyncTimestamp() {
	poc.LastSyncTimestamp = time.Now().Unix()
}

func (poc RacktablesOwnersCache) GetFQDNOwnerSubjects(fqdn string) []racktables.SubjectInfo {
	return poc.RacktablesOwners.GetAllFQDNOwners(fqdn)
}

func (poc RacktablesOwnersCache) GetMacroOwnerSubjects(macro string) []racktables.SubjectInfo {
	return poc.RacktablesOwners.GetAllMacroOwners(macro)
}

func (poc *RacktablesOwnersCache) StartPeriodicSyncing(periodSeconds int64) bool {
	if poc.PeriodSyncingIsOn {
		return false
	}

	poc.PeriodSyncingIsOn = true

	syncer := func(ctx context.Context) {
		for {
			select {
			case <-ctx.Done():
				return
			default:
				if time.Now().Unix()-poc.LastSyncTimestamp > periodSeconds {
					poc.Sync()
				} else {
					time.Sleep(1 * time.Second)
				}
			}
		}
	}
	poc.PeriodicSyncingContext, poc.ContextCancelFunc = context.WithCancel(context.Background())

	go syncer(poc.PeriodicSyncingContext)

	return true
}

func (poc *RacktablesOwnersCache) StopPeriodicSyncing() bool {
	if poc.PeriodSyncingIsOn {
		poc.ContextCancelFunc()
		poc.PeriodSyncingIsOn = false
		return true
	} else {
		return false
	}
}
