package racktables

import (
	"crypto/tls"
	"fmt"
	"strings"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/certifi"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/nop"
)

type RacktablesClient struct {
	Client resty.Client
	Logger log.Logger
}

type RacktablesResponseL123 struct {
	IP            string  `json:"IP"`
	Network       *string `json:"Network"`
	Location      *string `json:"Location"`
	VLAN          *string `json:"VLAN"`
	MAC           *string `json:"MAC"`
	Switch        *string `json:"Switch"`
	Port          *string `json:"Port"`
	PortTimestamp *int    `json:"Port_timestamp"`
}

type RacktablesNetsByProject struct {
	Network      string
	Macro        *string
	VLAN         *string
	Project      *string
	Responsibles *string
	DC           *string
}

func NewHTTPClient() (*resty.Client, error) {
	certPool, err := certifi.NewCertPool()
	if err != nil {
		return nil, err
	}

	httpClient := resty.New().
		SetRetryCount(2).
		SetHeader("User-Agent", "ya_resolve <security@yandex-team.ru>").
		SetTLSClientConfig(&tls.Config{RootCAs: certPool})

	return httpClient, nil
}

func NewRacktablesClient(options ...Option) (*RacktablesClient, error) {
	httpClient, err := NewHTTPClient()
	if err != nil {
		return nil, err
	}

	racktablesClient := RacktablesClient{
		Client: *httpClient,
		Logger: &nop.Logger{},
	}

	for _, opt := range options {
		opt(&racktablesClient)
	}

	return &racktablesClient, nil
}

func (p *RacktablesClient) GetMACByIP(ip string) (*string, error) {

	var result RacktablesResponseL123

	resp, err := p.Client.R().
		SetQueryParams(map[string]string{
			"q":    ip,
			"json": "1",
		}).
		SetResult(&result).
		Get("https://racktables.yandex.net/export/get-l123.php")

	if err != nil {
		return nil, err
	}

	if !resp.IsSuccess() {
		return nil, fmt.Errorf("RacktablesClient::GetMACByIP. !resp.IsSuccess(). Status code: %d", resp.StatusCode())
	}

	if result.MAC == nil {
		return nil, fmt.Errorf("RacktablesClient::GetMACByIP. result.MAC == nil. Unknown for racktables ip address")
	}

	return result.MAC, nil
}

func (p *RacktablesClient) GetNetworkByIP(ip string) (*string, error) {

	var result RacktablesResponseL123

	resp, err := p.Client.R().
		SetQueryParams(map[string]string{
			"q":    ip,
			"json": "1",
		}).
		SetResult(&result).
		Get("https://racktables.yandex.net/export/get-l123.php")

	if err != nil {
		return nil, err
	}

	if !resp.IsSuccess() {
		return nil, fmt.Errorf("RacktablesClient::GetNetworkByIP. !resp.IsSuccess(). Status code: %d", resp.StatusCode())
	}

	if result.Network == nil {
		return nil, fmt.Errorf("RacktablesClient::GetNetworkByIP. result.Network == nil. Unknown for racktables ip address")
	}

	if !strings.Contains(*result.Network, "/") {
		return nil, fmt.Errorf("RacktablesClient::GetNetworkByIP. result.Network is not network: %s", *result.Network)
	}

	return result.Network, nil
}

func (p *RacktablesClient) GetMacroByIP(ip string) (*string, error) {

	resp, err := p.Client.R().
		SetQueryParams(map[string]string{
			"ip": ip,
		}).
		Get("https://racktables.yandex.net/export/get-net-by-ip.php")

	if err != nil {
		return nil, err
	}

	if !resp.IsSuccess() {
		return nil, fmt.Errorf("RacktablesClient::GetMacroByIP. !resp.IsSuccess(). Status code: %d", resp.StatusCode())
	}

	respBody := string(resp.Body())
	parts := strings.Split(respBody, "\t")

	if len(parts) != 3 {
		return nil, fmt.Errorf("RacktablesClient::GetMacroByIP. len(parts) != 3")
	}

	macro := strings.TrimSpace(parts[2])

	if !strings.HasPrefix(macro, "_") || !strings.HasSuffix(macro, "_") {
		errMsg := fmt.Sprintf("RacktablesClient::GetMacroByIP. Prefix / Suffix mismatch in \"%s\".", macro)
		p.Logger.Warnf(errMsg)
		return nil, fmt.Errorf(errMsg)
	}

	return &macro, nil
}

func (p *RacktablesClient) FetchNetsByProject() (map[string]RacktablesNetsByProject, error) {

	resp, err := p.Client.R().Get("https://racktables.yandex.net/export/nets-by-project.php")

	if err != nil {
		return nil, err
	}

	if !resp.IsSuccess() {
		return nil, fmt.Errorf("RacktablesClient::FetchNetsByProject. !resp.IsSuccess(). Status code: %d", resp.StatusCode())
	}

	mapNetsByProject := map[string]RacktablesNetsByProject{}

	respBody := string(resp.Body())
	lines := strings.Split(respBody, "\n")

	for _, line := range lines {
		parts := strings.Split(line, "\t")
		if len(parts) != 6 {
			p.Logger.Infof("RacktablesClient::FetchNetsByProject. Incomplete line: %s", line)
			continue
		}

		if !strings.Contains(parts[0], "/") {
			p.Logger.Infof("RacktablesClient::FetchNetsByProject. Not a network: %s in %s. skip", parts[0], line)
		}

		netsByProject := RacktablesNetsByProject{
			Network: parts[0],
		}

		if parts[1] != "" {
			netsByProject.Macro = &parts[1]
		}

		if parts[2] != "" {
			netsByProject.VLAN = &parts[2]
		}

		if parts[3] != "" {
			netsByProject.Project = &parts[3]
		}

		if parts[4] != "" {
			netsByProject.Responsibles = &parts[4]
		}

		if parts[5] != "" {
			netsByProject.DC = &parts[5]
		}

		mapNetsByProject[parts[0]] = netsByProject
	}

	return mapNetsByProject, nil
}
