package resolver

import (
	"fmt"
	"net"
	"regexp"
	"strconv"
	"strings"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/nop"
	"a.yandex-team.ru/security/ya_resolve/internal/abc"
	"a.yandex-team.ru/security/ya_resolve/internal/bot"
	"a.yandex-team.ru/security/ya_resolve/internal/macros"
	"a.yandex-team.ru/security/ya_resolve/internal/puncher"
	"a.yandex-team.ru/security/ya_resolve/internal/racktables"
	"a.yandex-team.ru/security/ya_resolve/internal/racktables/netsbyprojectcache"
	RacktablesOwnersCache "a.yandex-team.ru/security/ya_resolve/internal/racktables/ownerscache"
	"a.yandex-team.ru/security/ya_resolve/internal/staff"
	"a.yandex-team.ru/security/ya_resolve/internal/walle"
	walleCache "a.yandex-team.ru/security/ya_resolve/internal/walle/cache"
)

type (
	ResStatus string
	ValueType string
	ResName   string
)

const (
	TypeTVM               ValueType = "tvm"
	TypeMacro             ValueType = "macro"
	TypeFQDN              ValueType = "fqdn"
	TypeIPAddress         ValueType = "ip_address"
	TypeMACAddress        ValueType = "mac_address"
	TypeBotInstanceNumber ValueType = "bot_instance_number"
	TypeDepartment        ValueType = "department"
	TypeService           ValueType = "service"
	TypeServiceRole       ValueType = "servicerole"
	TypePerson            ValueType = "person"
	TypeWikiGroup         ValueType = "wiki"
	TypeServiceID         ValueType = "service_id"
	TypeNetwork           ValueType = "network"
	TypeUnknown           ValueType = "unknown"

	ResStatusProcessing ResStatus = "processing"
	ResStatusSuccess    ResStatus = "success"
	ResStatusFail       ResStatus = "fail"

	ResolverRacktablesNetworkResps ResName = "racktables_network_resps"
	ResolverRacktablesNetwork      ResName = "racktables_network"
	ResolverOwnersFQDN             ResName = "racktables_owners_fqdn"
	ResolverOwnersMacro            ResName = "racktables_owners_macro"
	ResolverTVM                    ResName = "tvm"
	ResolverRacktablesIP2MAC       ResName = "racktables_ip2mac"
	ResolverRacktablesIP2Macro     ResName = "ip2macro_racktables"
	ResolverIP2Macro               ResName = "ip2macro"
	ResolverHBFIP2Macro            ResName = "ip2macro_hbf"
	ResolverBot                    ResName = "bot"
	ResolverABCResource            ResName = "abc_resource"
	ResolverPerson                 ResName = "person"
	ResolverService                ResName = "service"
	ResolverServiceRole            ResName = "servicerole"
	ResolverDepartment             ResName = "department"
	ResolverWikiGroup              ResName = "wiki"
	ResolverGeneral                ResName = "general"
	ResolverWalle                  ResName = "walle"
	ResolverServiceID2Members      ResName = "service_id_2_members"
	ResolverDNSIP2FQDN             ResName = "dns_ip_2_fqdn"
	ResolverDNSFQDN2IP             ResName = "dns_fqdn_2_ip"
	ResolverIP2PuncherResps        ResName = "ip_2_puncher_responsibles"
	ResolverFQDN2PuncherResps      ResName = "fqdn_2_puncher_responsibles"
	ResolverMacro2PuncherResps     ResName = "macro_2_puncher_responsibles"

	ReasonHeadOfDepartmentOfFiredPerson string = "Head of departmet of fired person"
	ReasonRobotOwner                    string = "Owner of robot"

	DefaultResolverMaxDepth = 7
)

type FlowEntry struct {
	Resolution string  `json:"resolution"`
	Resolver   ResName `json:"resolver"`
	Value      string  `json:"value"`
}

type Resolution struct {
	Flow   []FlowEntry `json:"flow"`
	Status ResStatus   `json:"status"`
	Type   ValueType   `json:"type"`
	Value  string      `json:"value"`
	Weight int         `json:"weight"`
}

func (r Resolution) IsServiceID() bool {
	if _, err := strconv.Atoi(r.Value); err != nil {
		return false
	}
	return r.Type == TypeServiceID || r.Type == TypeUnknown
}

func (r Resolution) IsNetwork() bool {
	if !strings.Contains(r.Value, "/") {
		return false
	}
	return r.Type == TypeNetwork || r.Type == TypeUnknown
}

func (r Resolution) IsMacro() bool {
	if r.Type != TypeMacro && r.Type == TypeUnknown {
		return false
	}
	if matched, err := regexp.MatchString("^[A-Z0-9_]*$", r.Value); err != nil || !matched {
		return false
	}
	return true
}

func (r Resolution) IsFQDN() bool {
	if r.Type != TypeFQDN && r.Type != TypeUnknown {
		return false
	}
	parts := strings.Split(r.Value, ".")
	if len(parts) <= 1 {
		return false
	}
	// check tld
	if matched, err := regexp.MatchString("^[a-zA-Z]*$", parts[len(parts)-1]); err != nil || !matched {
		return false
	}
	return true
}

func (r Resolution) IsPerson() bool {
	return r.Type == TypePerson || r.Type == TypeUnknown
}

func (r Resolution) IsServiceRole() bool {
	return r.Type == TypeServiceRole || r.Type == TypeUnknown
}

func (r Resolution) IsWikiGroup() bool {
	return r.Type == TypeWikiGroup || r.Type == TypeUnknown
}

func (r Resolution) IsDepartment() bool {
	return r.Type == TypeDepartment || r.Type == TypeUnknown
}

func (r Resolution) IsService() bool {
	return r.Type == TypeService || r.Type == TypeUnknown
}

func (r Resolution) IsIPAddress() bool {
	if r.Type != TypeIPAddress && r.Type != TypeUnknown {
		return false
	}
	ip := net.ParseIP(r.Value)
	return ip != nil
}

func (r Resolution) IsMACAddress() bool {
	if r.Type != TypeMACAddress && r.Type != TypeUnknown {
		return false
	}
	if matched, err := regexp.MatchString("^([0-9A-Fa-f]{2}(:|-|\\.)?){5}([0-9A-Fa-f]{2})$", r.Value); err != nil || !matched {
		return false
	}
	return true
}

func (r Resolution) IsBotInstanceNumber() bool {
	return r.Type == TypeBotInstanceNumber || r.Type == TypeUnknown
}

func (r Resolution) IsTVM() bool {
	return r.Type == TypeTVM || r.Type == TypeUnknown
}

func (r Resolution) GetDepth() int {
	return len(r.Flow)
}

func UnsupportedResolution(resolution Resolution) Resolution {
	res := Resolution{
		Flow: append(resolution.Flow, FlowEntry{
			Resolution: "Unsupported resolution",
			Resolver:   ResolverGeneral,
			Value:      resolution.Value,
		}),
		Status: ResStatusFail,
		Type:   resolution.Type,
		Value:  resolution.Value,
	}
	return res
}

func MaxDepthExceededResolution(resolution Resolution) Resolution {
	return Resolution{
		Flow: append(resolution.Flow, FlowEntry{
			Resolution: "Max depth exceeded",
			Resolver:   ResolverGeneral,
			Value:      resolution.Value,
		}),
		Status: ResStatusFail,
		Type:   resolution.Type,
		Value:  resolution.Value,
	}
}

func RecursionBlockResolution(resolution Resolution) Resolution {
	return Resolution{
		Flow: append(resolution.Flow, FlowEntry{
			Resolution: "Recursion detected",
			Resolver:   ResolverGeneral,
			Value:      resolution.Value,
		}),
		Status: ResStatusFail,
		Type:   resolution.Type,
		Value:  resolution.Value,
	}
}

type Resolver struct {
	ABC                   abc.ABC
	Bot                   bot.BotClient
	Racktables            racktables.RacktablesClient
	RacktablesOwnersCache RacktablesOwnersCache.RacktablesOwnersCache
	NetsByProjectCache    netsbyprojectcache.NetsByProjectCache
	Staff                 staff.Staff
	Walle                 walle.Client
	WalleCache            walleCache.WalleCache
	HBFClient             macros.HBFClient
	MacrosCache           macros.MacrosCache
	Puncher               puncher.Client
	ResolverMaxDepth      int
	Logger                log.Logger
}

func NewResolver(
	abcClient abc.ABC,
	staffClient staff.Staff,
	racktablesClient racktables.RacktablesClient,
	botClient bot.BotClient,
	walleClient walle.Client,
	hbfClient macros.HBFClient,
	puncherClient puncher.Client,
	options ...Option,
) Resolver {

	r := Resolver{
		ABC:                   abcClient,
		Bot:                   botClient,
		Racktables:            racktablesClient,
		RacktablesOwnersCache: RacktablesOwnersCache.NewRacktablesOwnersCache(racktablesClient),
		NetsByProjectCache:    netsbyprojectcache.New(racktablesClient),
		Staff:                 staffClient,
		Walle:                 walleClient,
		WalleCache:            walleCache.NewCache(walleClient),
		HBFClient:             hbfClient,
		Puncher:               puncherClient,
		MacrosCache:           macros.NewCache(hbfClient, macros.CacheWithLogger(hbfClient.Logger)),
		ResolverMaxDepth:      DefaultResolverMaxDepth,
		Logger:                &nop.Logger{},
	}

	r.RacktablesOwnersCache.Sync()
	r.RacktablesOwnersCache.StartPeriodicSyncing(60 * 10)

	r.NetsByProjectCache.Sync()
	r.NetsByProjectCache.StartPeriodicSyncing(60 * 10)

	r.WalleCache.Sync()
	r.WalleCache.StartPeriodicSyncing(60 * 10)

	r.MacrosCache.UpdateCache()
	r.MacrosCache.StartCron()

	for _, opt := range options {
		opt(&r)
	}

	return r
}

func (r Resolver) ResolveMacro(resolution Resolution) []Resolution {
	var processedResolutions []Resolution

	value := strings.Trim(resolution.Value, "_")

	subjectInfoList := r.RacktablesOwnersCache.GetMacroOwnerSubjects(value)

	if len(subjectInfoList) == 0 {
		flowEntry := FlowEntry{
			Resolution: "No owners for macro found",
			Resolver:   ResolverOwnersMacro,
			Value:      value,
		}
		res := Resolution{
			Flow:   append(resolution.Flow, flowEntry),
			Status: ResStatusFail,
			Type:   TypeMacro,
			Value:  value,
		}
		processedResolutions = append(processedResolutions, res)
		return processedResolutions
	}

	// prepare good flow
	flowEntry := FlowEntry{
		Resolution: "Owner of macro by racktables owners",
		Resolver:   ResolverOwnersMacro,
		Value:      value,
	}
	flow := append(resolution.Flow, flowEntry)

	// prepare bad flow and bad resolution
	badFlowEntry := FlowEntry{
		Resolution: "Unknown subject type for racktables owners resolver",
		Resolver:   ResolverOwnersMacro,
		Value:      value,
	}
	badFlow := append(resolution.Flow, badFlowEntry)

	// iterate ower fqdn owners
	for _, subjectInfo := range subjectInfoList {

		res := Resolution{
			Flow:   flow,
			Status: ResStatusProcessing,
			Type:   TypeUnknown,
			Value:  subjectInfo.Subj,
		}
		switch subjectInfo.SubjType {

		case racktables.SubjTypeService:
			res.Type = TypeService
			processedResolutions = append(processedResolutions, r.ResolveImp(res)...)

		case racktables.SubjTypeServiceRole:
			res.Type = TypeServiceRole
			processedResolutions = append(processedResolutions, r.ResolveImp(res)...)

		case racktables.SubjTypeDepartment:
			res.Type = TypeDepartment
			processedResolutions = append(processedResolutions, r.ResolveImp(res)...)

		case racktables.SubjTypeUser:
			res.Type = TypePerson
			processedResolutions = append(processedResolutions, r.ResolveImp(res)...)

		default:
			badRes := Resolution{
				Flow:   badFlow,
				Status: ResStatusFail,
				Type:   TypeUnknown,
				Value:  subjectInfo.Subj,
			}
			processedResolutions = append(processedResolutions, badRes)
		}

	}

	return processedResolutions
}

func (r Resolver) ResolveFQDNWalle(resolution Resolution) []Resolution {
	var processedResolutions []Resolution

	host, err := r.WalleCache.GetHostInfoByFQDN(resolution.Value)
	if err != nil {
		flowEntry := FlowEntry{
			Resolution: fmt.Sprintf("%v", err),
			Resolver:   ResolverWalle,
			Value:      resolution.Value,
		}
		res := Resolution{
			Flow:   append(resolution.Flow, flowEntry),
			Status: ResStatusFail,
			Type:   TypeFQDN,
			Value:  resolution.Value,
		}
		processedResolutions = append(processedResolutions, res)
		return processedResolutions
	}

	if len(host.ProjectOwners) == 0 && len(host.StateAuthor) == 0 && len(host.StatusAuthor) == 0 {
		failMessage := fmt.Sprintf("no owners, not status and state authors. only provisioner presented: %s", host.ProjectProvisioner)
		r.Logger.Warnf(failMessage)

		flowEntry := FlowEntry{
			Resolution: failMessage,
			Resolver:   ResolverWalle,
			Value:      resolution.Value,
		}
		res := Resolution{
			Flow:   append(resolution.Flow, flowEntry),
			Status: ResStatusFail,
			Type:   TypeFQDN,
			Value:  resolution.Value,
		}
		processedResolutions = append(processedResolutions, res)
		return processedResolutions
	}

	var resps []string
	resps = append(resps, host.ProjectOwners...)
	if len(resps) == 0 {
		resps = append(resps, host.StateAuthor)
	}
	if len(resps) == 0 {
		resps = append(resps, host.StatusAuthor)
	}

	flowEntry := FlowEntry{
		Resolution: fmt.Sprintf("Owner of host %s. Project: %s", host.Name, host.ProjectID),
		Resolver:   ResolverWalle,
		Value:      resolution.Value,
	}
	flow := append(resolution.Flow, flowEntry)
	for _, resp := range resps {
		res := Resolution{
			Flow:   flow,
			Status: ResStatusProcessing,
			Type:   TypePerson,
			Value:  resp,
		}
		processedResolutions = append(processedResolutions, r.ResolveImp(res)...)
	}

	return processedResolutions
}

func (r Resolver) ResolveFQDNRacktablesOwners(resolution Resolution) []Resolution {
	var processedResolutions []Resolution

	subjectInfoList := r.RacktablesOwnersCache.GetFQDNOwnerSubjects(resolution.Value)

	if len(subjectInfoList) == 0 {
		flowEntry := FlowEntry{
			Resolution: "No owners for fqdn found",
			Resolver:   ResolverOwnersFQDN,
			Value:      resolution.Value,
		}
		res := Resolution{
			Flow:   append(resolution.Flow, flowEntry),
			Status: ResStatusFail,
			Type:   TypeFQDN,
			Value:  resolution.Value,
		}
		processedResolutions = append(processedResolutions, res)
		return processedResolutions
	}

	// prepare good flow
	flowEntry := FlowEntry{
		Resolution: "Owner of fqdn by racktables owners",
		Resolver:   ResolverOwnersFQDN,
		Value:      resolution.Value,
	}
	flow := append(resolution.Flow, flowEntry)

	// prepare bad flow and bad resolution
	badFlowEntry := FlowEntry{
		Resolution: "Unknown subject type for racktables owners resolver",
		Resolver:   ResolverOwnersFQDN,
		Value:      resolution.Value,
	}
	badFlow := append(resolution.Flow, badFlowEntry)

	// iterate ower fqdn owners
	for _, subjectInfo := range subjectInfoList {

		res := Resolution{
			Flow:   flow,
			Status: ResStatusProcessing,
			Type:   TypeUnknown,
			Value:  subjectInfo.Subj,
		}
		switch subjectInfo.SubjType {

		case racktables.SubjTypeService:
			res.Type = TypeService
			processedResolutions = append(processedResolutions, r.ResolveImp(res)...)

		case racktables.SubjTypeServiceRole:
			res.Type = TypeServiceRole
			processedResolutions = append(processedResolutions, r.ResolveImp(res)...)

		case racktables.SubjTypeDepartment:
			res.Type = TypeDepartment
			processedResolutions = append(processedResolutions, r.ResolveImp(res)...)

		case racktables.SubjTypeUser:
			res.Type = TypePerson
			processedResolutions = append(processedResolutions, r.ResolveImp(res)...)

		default:
			badRes := Resolution{
				Flow:   badFlow,
				Status: ResStatusFail,
				Type:   TypeUnknown,
				Value:  subjectInfo.Subj,
			}
			processedResolutions = append(processedResolutions, badRes)
		}

	}

	return processedResolutions
}

func (r Resolver) ResolvePerson(resolution Resolution) []Resolution {

	var processedResolutions []Resolution

	person, err := r.Staff.GetUserInfo(resolution.Value)

	switch {

	case err != nil:
		flowEntry := FlowEntry{
			Resolution: fmt.Sprint("Error: ", err),
			Resolver:   ResolverPerson,
			Value:      resolution.Value,
		}
		res := Resolution{
			Flow:   append(resolution.Flow, flowEntry),
			Status: ResStatusFail,
			Type:   resolution.Type,
			Value:  resolution.Value,
		}
		processedResolutions = append(processedResolutions, res)

	case !person.IsFired() && !person.IsRobot():
		flowEntry := FlowEntry{
			Resolution: "Resolved",
			Resolver:   ResolverPerson,
			Value:      resolution.Value,
		}
		res := Resolution{
			Flow:   append(resolution.Flow, flowEntry),
			Status: ResStatusSuccess,
			Type:   TypePerson,
			Value:  resolution.Value,
		}
		processedResolutions = append(processedResolutions, r.ResolveImp(res)...)

	case person.IsRobot():
		flowEntry := FlowEntry{
			Resolution: ReasonRobotOwner,
			Resolver:   ResolverPerson,
			Value:      resolution.Value,
		}
		flow := append(resolution.Flow, flowEntry)
		for _, robotOwner := range person.RobotOwners {
			res := Resolution{
				Flow:   flow,
				Status: ResStatusProcessing,
				Type:   TypePerson,
				Value:  robotOwner.Person.Login,
			}
			processedResolutions = append(processedResolutions, r.ResolveImp(res)...)
		}

	case person.IsFired():
		flowEntry := FlowEntry{
			Resolution: ReasonHeadOfDepartmentOfFiredPerson,
			Resolver:   ResolverPerson,
			Value:      resolution.Value,
		}
		flow := append(resolution.Flow, flowEntry)
		for _, departmentHead := range person.DepartmentGroup.Department.Heads {
			res := Resolution{
				Flow:   flow,
				Status: ResStatusProcessing,
				Type:   TypePerson,
				Value:  departmentHead.Person.Login,
			}
			processedResolutions = append(processedResolutions, r.ResolveImp(res)...)
		}

	default:
		flowEntry := FlowEntry{
			Resolution: "Unknown problem in resolving",
			Resolver:   ResolverPerson,
			Value:      resolution.Value,
		}
		res := Resolution{
			Flow:   append(resolution.Flow, flowEntry),
			Status: ResStatusFail,
			Type:   resolution.Type,
			Value:  resolution.Value,
		}
		processedResolutions = append(processedResolutions, res)

	}

	return processedResolutions
}

func (r Resolver) ResolveServiceRole(resolution Resolution) []Resolution {
	var processedResolutions []Resolution

	servicerolePersons, err := r.Staff.GetServiceRoleMembers(resolution.Value)

	switch {

	case err != nil:
		flowEntry := FlowEntry{
			Resolution: fmt.Sprint("Error: ", err),
			Resolver:   ResolverServiceRole,
			Value:      resolution.Value,
		}
		res := Resolution{
			Flow:   append(resolution.Flow, flowEntry),
			Status: ResStatusFail,
			Type:   resolution.Type,
			Value:  resolution.Value,
		}
		processedResolutions = append(processedResolutions, res)

	case len(servicerolePersons) == 0:
		flowEntry := FlowEntry{
			Resolution: "Empty servicerole",
			Resolver:   ResolverServiceRole,
			Value:      resolution.Value,
		}
		res := Resolution{
			Flow:   append(resolution.Flow, flowEntry),
			Status: ResStatusFail,
			Type:   resolution.Type,
			Value:  resolution.Value,
		}
		processedResolutions = append(processedResolutions, res)

	default:
		for _, person := range servicerolePersons {
			flowEntry := FlowEntry{
				Resolution: "Servicerole member",
				Resolver:   ResolverServiceRole,
				Value:      resolution.Value,
			}
			res := Resolution{
				Flow:   append(resolution.Flow, flowEntry),
				Status: ResStatusProcessing,
				Type:   TypePerson,
				Value:  person.Login,
			}
			processedResolutions = append(processedResolutions, r.ResolveImp(res)...)
		}

	}

	return processedResolutions
}

func (r Resolver) ResolveDepartment(resolution Resolution) []Resolution {
	var processedResolutions []Resolution

	departmentPersons, err := r.Staff.GetDepartmentMembers(resolution.Value)

	switch {

	case err != nil:
		flowEntry := FlowEntry{
			Resolution: fmt.Sprintf("Error: %v", err),
			Resolver:   ResolverDepartment,
			Value:      resolution.Value,
		}
		res := Resolution{
			Flow:   append(resolution.Flow, flowEntry),
			Status: ResStatusFail,
			Type:   resolution.Type,
			Value:  resolution.Value,
		}
		processedResolutions = append(processedResolutions, res)
		return processedResolutions

	case len(departmentPersons) == 0:
		flowEntry := FlowEntry{
			Resolution: "Empty department",
			Resolver:   ResolverDepartment,
			Value:      resolution.Value,
		}
		res := Resolution{
			Flow:   append(resolution.Flow, flowEntry),
			Status: ResStatusFail,
			Type:   resolution.Type,
			Value:  resolution.Value,
		}
		processedResolutions = append(processedResolutions, res)
		return processedResolutions

	default:
		for _, person := range departmentPersons {
			flowEntry := FlowEntry{
				Resolution: "Department member",
				Resolver:   ResolverDepartment,
				Value:      resolution.Value,
			}
			res := Resolution{
				Flow:   append(resolution.Flow, flowEntry),
				Status: ResStatusProcessing,
				Type:   TypePerson,
				Value:  person.Login,
			}
			processedResolutions = append(processedResolutions, r.ResolveImp(res)...)
		}

	}

	return processedResolutions
}

func (r Resolver) ResolveServiceID(resolution Resolution) []Resolution {
	var processedResolutions []Resolution

	serviceID, err := strconv.Atoi(resolution.Value)

	if err != nil {
		flowEntry := FlowEntry{
			Resolution: fmt.Sprintf("Error in parsing value as int: %v", err),
			Resolver:   ResolverServiceID2Members,
			Value:      resolution.Value,
		}
		res := Resolution{
			Flow:   append(resolution.Flow, flowEntry),
			Status: ResStatusFail,
			Type:   TypeServiceID,
			Value:  resolution.Value,
		}
		processedResolutions = append(processedResolutions, res)
		return processedResolutions
	}

	// resolve service by id
	ABCResolutions, err := r.ABC.ResolveServiceIDToMembers(serviceID)

	// return fail if error on resolving service slug
	if err != nil {
		flowEntry := FlowEntry{
			Resolution: fmt.Sprintf("Error: %v", err),
			Resolver:   ResolverServiceID2Members,
			Value:      resolution.Value,
		}
		res := Resolution{
			Flow:   append(resolution.Flow, flowEntry),
			Status: ResStatusFail,
			Type:   TypeServiceID,
			Value:  resolution.Value,
		}
		processedResolutions = append(processedResolutions, res)
		return processedResolutions
	}

	// convert abc.ABCResolution to resolver.Resolution
	for _, ABCResolution := range ABCResolutions {
		flowEntry := FlowEntry{
			Resolution: ABCResolution.ResolveFlow,
			Resolver:   ResolverServiceID2Members,
			Value:      resolution.Value,
		}
		res := Resolution{
			Flow:   append(resolution.Flow, flowEntry),
			Status: ResStatusProcessing,
			Type:   TypePerson,
			Value:  ABCResolution.User,
		}
		processedResolutions = append(processedResolutions, r.ResolveImp(res)...)
	}

	return processedResolutions
}

func (r Resolver) ResolveService(resolution Resolution) []Resolution {
	var processedResolutions []Resolution

	// fix service name
	// for example: svc_security -> security
	value := strings.TrimPrefix(resolution.Value, "svc_")

	// resolve service by slug
	ABCResolutions, err := r.ABC.ResolveSlug(value)

	// return fail if error on resolving service slug
	if err != nil {
		flowEntry := FlowEntry{
			Resolution: fmt.Sprintf("Error: %v", err),
			Resolver:   ResolverService,
			Value:      value,
		}
		res := Resolution{
			Flow:   append(resolution.Flow, flowEntry),
			Status: ResStatusFail,
			Type:   TypeService,
			Value:  value,
		}
		processedResolutions = append(processedResolutions, res)
		return processedResolutions
	}

	// convert abc.ABCResolution to resolver.Resolution
	for _, ABCResolution := range ABCResolutions {
		flowEntry := FlowEntry{
			Resolution: ABCResolution.ResolveFlow,
			Resolver:   ResolverService,
			Value:      value,
		}
		res := Resolution{
			Flow:   append(resolution.Flow, flowEntry),
			Status: ResStatusProcessing,
			Type:   TypePerson,
			Value:  ABCResolution.User,
		}
		processedResolutions = append(processedResolutions, r.ResolveImp(res)...)
	}

	return processedResolutions
}

func (r Resolver) ResolveIPAddressToNetwork(resolution Resolution) []Resolution {
	var processedResolutions []Resolution

	network, err := r.Racktables.GetNetworkByIP(resolution.Value)

	if err != nil {
		flowEntry := FlowEntry{
			Resolution: fmt.Sprintf("error: %v", err),
			Resolver:   ResolverRacktablesNetwork,
			Value:      resolution.Value,
		}
		res := Resolution{
			Flow:   append(resolution.Flow, flowEntry),
			Status: ResStatusFail,
			Type:   TypeNetwork,
			Value:  resolution.Value,
		}
		processedResolutions = append(processedResolutions, res)
		return processedResolutions
	}

	flowEntry := FlowEntry{
		Resolution: fmt.Sprintf("Network for ip (%s)", resolution.Value),
		Resolver:   ResolverRacktablesNetwork,
		Value:      resolution.Value,
	}
	res := Resolution{
		Flow:   append(resolution.Flow, flowEntry),
		Status: ResStatusProcessing,
		Type:   TypeNetwork,
		Value:  *network,
	}
	processedResolutions = append(processedResolutions, r.ResolveImp(res)...)
	return processedResolutions
}

func (r Resolver) ResolveIPAddressToMACAddress(resolution Resolution) []Resolution {
	var processedResolutions []Resolution

	macAddress, err := r.Racktables.GetMACByIP(resolution.Value)

	if err != nil {
		flowEntry := FlowEntry{
			Resolution: fmt.Sprintf("error: %v", err),
			Resolver:   ResolverRacktablesIP2MAC,
			Value:      resolution.Value,
		}
		res := Resolution{
			Flow:   append(resolution.Flow, flowEntry),
			Status: ResStatusFail,
			Type:   TypeIPAddress,
			Value:  resolution.Value,
		}
		processedResolutions = append(processedResolutions, res)
		return processedResolutions
	}

	flowEntry := FlowEntry{
		Resolution: fmt.Sprintf("Mac address of ip (%s)", resolution.Value),
		Resolver:   ResolverRacktablesIP2MAC,
		Value:      resolution.Value,
	}
	res := Resolution{
		Flow:   append(resolution.Flow, flowEntry),
		Status: ResStatusProcessing,
		Type:   TypeMACAddress,
		Value:  *macAddress,
	}
	processedResolutions = append(processedResolutions, r.ResolveImp(res)...)
	return processedResolutions
}

func (r Resolver) GetMacroForIPWithHBFCache(resolution Resolution) ([]string, error) {

	ip := net.ParseIP(resolution.Value)
	if ip == nil {
		return nil, fmt.Errorf("not an ip")
	}

	projectID, err := macros.ExtractProjectID(ip)
	if err != nil {
		return nil, fmt.Errorf("error on extract project: %v", err)
	}

	macros, ok := r.MacrosCache.GetMacrosByProjectID(*projectID)
	if !ok {
		return nil, fmt.Errorf("macro not found for projectID: %x", *projectID)
	}

	return macros, nil
}

func (r Resolver) GetMacroForIPWithRacktables(resolution Resolution) ([]string, error) {

	macro, err := r.Racktables.GetMacroByIP(resolution.Value)
	if err != nil {
		return nil, err
	}

	macros := []string{*macro}

	return macros, err
}

func (r Resolver) ResolveIPAddressToMacro(resolution Resolution) []Resolution {
	var processedResolutions []Resolution

	var err error
	var macros []string
	resolver := ResolverHBFIP2Macro

	macros, err = r.GetMacroForIPWithHBFCache(resolution)
	if err != nil {
		macro, err := r.Racktables.GetMacroByIP(resolution.Value)
		if err == nil {
			macros = []string{*macro}
			resolver = ResolverRacktablesIP2Macro
		}
	}

	if err != nil {
		flowEntry := FlowEntry{
			Resolution: fmt.Sprintf("error: %v", err),
			Resolver:   ResolverIP2Macro,
			Value:      resolution.Value,
		}
		res := Resolution{
			Flow:   append(resolution.Flow, flowEntry),
			Status: ResStatusFail,
			Type:   TypeIPAddress,
			Value:  resolution.Value,
		}
		processedResolutions = append(processedResolutions, res)
		return processedResolutions
	}

	flowEntry := FlowEntry{
		Resolution: "Macro for this ip address",
		Resolver:   resolver,
		Value:      resolution.Value,
	}
	for _, macro := range macros {
		res := Resolution{
			Flow:   append(resolution.Flow, flowEntry),
			Status: ResStatusProcessing,
			Type:   TypeMacro,
			Value:  macro,
		}
		processedResolutions = append(processedResolutions, r.ResolveImp(res)...)
	}
	return processedResolutions
}

func (r Resolver) ResolveIPToFQDN(resolution Resolution) []Resolution {
	var processedResolutions []Resolution

	fqdns, err := net.LookupAddr(resolution.Value)

	if err != nil {
		flowEntry := FlowEntry{
			Resolution: fmt.Sprintf("error: %v", err),
			Resolver:   ResolverDNSIP2FQDN,
			Value:      resolution.Value,
		}
		res := Resolution{
			Flow:   append(resolution.Flow, flowEntry),
			Status: ResStatusFail,
			Type:   TypeIPAddress,
			Value:  resolution.Value,
		}
		processedResolutions = append(processedResolutions, res)
		return processedResolutions
	}

	flowEntry := FlowEntry{
		Resolution: "ip to fqdn lookup result",
		Resolver:   ResolverDNSIP2FQDN,
		Value:      resolution.Value,
	}
	for _, fqdn := range fqdns {
		res := Resolution{
			Flow:   append(resolution.Flow, flowEntry),
			Status: ResStatusProcessing,
			Type:   TypeFQDN,
			Value:  strings.TrimSuffix(fqdn, "."),
		}
		processedResolutions = append(processedResolutions, r.ResolveImp(res)...)
	}
	return processedResolutions
}

func (r Resolver) ResolveIPToPuncherResps(resolution Resolution) []Resolution {
	var processedResolutions []Resolution

	persons, err := r.Puncher.GetResponsibles(resolution.Value)

	if err != nil {
		flowEntry := FlowEntry{
			Resolution: fmt.Sprintf("error: %v", err),
			Resolver:   ResolverIP2PuncherResps,
			Value:      resolution.Value,
		}
		res := Resolution{
			Flow:   append(resolution.Flow, flowEntry),
			Status: ResStatusFail,
			Type:   TypeIPAddress,
			Value:  resolution.Value,
		}
		processedResolutions = append(processedResolutions, res)
		return processedResolutions
	}

	flowEntry := FlowEntry{
		Resolution: "ip to puncher resps result",
		Resolver:   ResolverIP2PuncherResps,
		Value:      resolution.Value,
	}
	for _, person := range persons {
		res := Resolution{
			Flow:   append(resolution.Flow, flowEntry),
			Status: ResStatusProcessing,
			Type:   TypePerson,
			Value:  person,
		}
		processedResolutions = append(processedResolutions, r.ResolveImp(res)...)
	}
	return processedResolutions
}

func (r Resolver) ResolveFQDNToPuncherResps(resolution Resolution) []Resolution {
	var processedResolutions []Resolution

	persons, err := r.Puncher.GetResponsibles(resolution.Value)

	if err != nil {
		flowEntry := FlowEntry{
			Resolution: fmt.Sprintf("error: %v", err),
			Resolver:   ResolverFQDN2PuncherResps,
			Value:      resolution.Value,
		}
		res := Resolution{
			Flow:   append(resolution.Flow, flowEntry),
			Status: ResStatusFail,
			Type:   TypeIPAddress,
			Value:  resolution.Value,
		}
		processedResolutions = append(processedResolutions, res)
		return processedResolutions
	}

	flowEntry := FlowEntry{
		Resolution: "fqdn to puncher resps result",
		Resolver:   ResolverFQDN2PuncherResps,
		Value:      resolution.Value,
	}
	for _, person := range persons {
		res := Resolution{
			Flow:   append(resolution.Flow, flowEntry),
			Status: ResStatusProcessing,
			Type:   TypePerson,
			Value:  person,
		}
		processedResolutions = append(processedResolutions, r.ResolveImp(res)...)
	}
	return processedResolutions
}

func (r Resolver) ResolveMacroToPuncherResps(resolution Resolution) []Resolution {
	var processedResolutions []Resolution

	persons, err := r.Puncher.GetResponsibles(resolution.Value)

	if err != nil {
		flowEntry := FlowEntry{
			Resolution: fmt.Sprintf("error: %v", err),
			Resolver:   ResolverMacro2PuncherResps,
			Value:      resolution.Value,
		}
		res := Resolution{
			Flow:   append(resolution.Flow, flowEntry),
			Status: ResStatusFail,
			Type:   TypeIPAddress,
			Value:  resolution.Value,
		}
		processedResolutions = append(processedResolutions, res)
		return processedResolutions
	}

	flowEntry := FlowEntry{
		Resolution: "marco to puncher resps result",
		Resolver:   ResolverMacro2PuncherResps,
		Value:      resolution.Value,
	}
	for _, person := range persons {
		res := Resolution{
			Flow:   append(resolution.Flow, flowEntry),
			Status: ResStatusProcessing,
			Type:   TypePerson,
			Value:  person,
		}
		processedResolutions = append(processedResolutions, r.ResolveImp(res)...)
	}
	return processedResolutions
}

func (r Resolver) ResolveFQDNToIP(resolution Resolution) []Resolution {
	var processedResolutions []Resolution

	addrs, err := net.LookupHost(resolution.Value)

	if err != nil {
		flowEntry := FlowEntry{
			Resolution: fmt.Sprintf("error: %v", err),
			Resolver:   ResolverDNSFQDN2IP,
			Value:      resolution.Value,
		}
		res := Resolution{
			Flow:   append(resolution.Flow, flowEntry),
			Status: ResStatusFail,
			Type:   TypeFQDN,
			Value:  resolution.Value,
		}
		processedResolutions = append(processedResolutions, res)
		return processedResolutions
	}

	flowEntry := FlowEntry{
		Resolution: "fqdn to ip lookup result",
		Resolver:   ResolverDNSFQDN2IP,
		Value:      resolution.Value,
	}
	for _, addr := range addrs {
		res := Resolution{
			Flow:   append(resolution.Flow, flowEntry),
			Status: ResStatusProcessing,
			Type:   TypeIPAddress,
			Value:  addr,
		}
		processedResolutions = append(processedResolutions, r.ResolveImp(res)...)
	}
	return processedResolutions
}

func (r Resolver) ResolveNetwork(resolution Resolution) []Resolution {
	var processedResolutions []Resolution

	info, ok := r.NetsByProjectCache.GetInfoByNetwork(resolution.Value)

	if !ok {
		flowEntry := FlowEntry{
			Resolution: fmt.Sprintf("Unknown network: %s", resolution.Value),
			Resolver:   ResolverRacktablesNetwork,
			Value:      resolution.Value,
		}
		res := Resolution{
			Flow:   append(resolution.Flow, flowEntry),
			Status: ResStatusFail,
			Type:   TypeNetwork,
			Value:  resolution.Value,
		}
		processedResolutions = append(processedResolutions, res)
		return processedResolutions
	}

	if info.Responsibles == nil || *info.Responsibles == "" {
		flowEntry := FlowEntry{
			Resolution: fmt.Sprintf("No responsibles for network: %s", resolution.Value),
			Resolver:   ResolverRacktablesNetwork,
			Value:      resolution.Value,
		}
		res := Resolution{
			Flow:   append(resolution.Flow, flowEntry),
			Status: ResStatusFail,
			Type:   TypeNetwork,
			Value:  resolution.Value,
		}
		processedResolutions = append(processedResolutions, res)
		return processedResolutions
	}

	flowEntry := FlowEntry{
		Resolution: fmt.Sprintf("Responsible for network (%s)", resolution.Value),
		Resolver:   ResolverRacktablesNetworkResps,
		Value:      resolution.Value,
	}
	flow := append(resolution.Flow, flowEntry)

	resps := strings.Split(*info.Responsibles, ",")

	for _, resp := range resps {
		res := Resolution{
			Flow:   flow,
			Status: ResStatusProcessing,
		}
		switch {
		case strings.HasPrefix(resp, "%") && strings.HasSuffix(resp, "%"):
			res.Type = TypePerson
			res.Value = strings.Trim(resp, "%")
		case strings.HasPrefix(resp, "svc_"):
			res.Type = TypeServiceRole
			res.Value = resp
		default:
			res.Type = TypeDepartment
			res.Value = resp
		}
		processedResolutions = append(processedResolutions, r.ResolveImp(res)...)
	}
	return processedResolutions
}

func (r Resolver) ResolveMACAddress(resolution Resolution) []Resolution {
	var processedResolutions []Resolution

	botInstanceNumbers, err := r.Bot.GetBotInstanceNumbersByMACAddress(resolution.Value)

	if err != nil {
		flowEntry := FlowEntry{
			Resolution: fmt.Sprintf("error: %v", err),
			Resolver:   ResolverBot,
			Value:      resolution.Value,
		}
		res := Resolution{
			Flow:   append(resolution.Flow, flowEntry),
			Status: ResStatusFail,
			Type:   TypeMACAddress,
			Value:  resolution.Value,
		}
		processedResolutions = append(processedResolutions, res)
		return processedResolutions
	}

	flowEntry := FlowEntry{
		Resolution: fmt.Sprintf("Bot Instance Number for MAC (%s)", resolution.Value),
		Resolver:   ResolverBot,
		Value:      resolution.Value,
	}
	flow := append(resolution.Flow, flowEntry)
	for _, botInstanceNumber := range botInstanceNumbers {
		res := Resolution{
			Flow:   flow,
			Status: ResStatusProcessing,
			Type:   TypeBotInstanceNumber,
			Value:  botInstanceNumber,
		}
		processedResolutions = append(processedResolutions, r.ResolveImp(res)...)
	}
	return processedResolutions
}

func (r Resolver) ResolveWikiGroup(resolution Resolution) []Resolution {
	var processedResolutions []Resolution

	wikiGroupMembers, err := r.Staff.GetWikiGroupMembers(resolution.Value)

	switch {

	case err != nil:
		flowEntry := FlowEntry{
			Resolution: fmt.Sprintf("error: %v", err),
			Resolver:   ResolverWikiGroup,
			Value:      resolution.Value,
		}
		res := Resolution{
			Flow:   append(resolution.Flow, flowEntry),
			Status: ResStatusFail,
			Type:   resolution.Type,
			Value:  resolution.Value,
		}
		processedResolutions = append(processedResolutions, res)
		return processedResolutions

	case len(wikiGroupMembers) == 0:
		flowEntry := FlowEntry{
			Resolution: "Empty wiki group",
			Resolver:   ResolverWikiGroup,
			Value:      resolution.Value,
		}
		res := Resolution{
			Flow:   append(resolution.Flow, flowEntry),
			Status: ResStatusFail,
			Type:   resolution.Type,
			Value:  resolution.Value,
		}
		processedResolutions = append(processedResolutions, res)
		return processedResolutions

	default:
		for _, person := range wikiGroupMembers {
			flowEntry := FlowEntry{
				Resolution: "Wiki group member",
				Resolver:   ResolverWikiGroup,
				Value:      resolution.Value,
			}
			res := Resolution{
				Flow:   append(resolution.Flow, flowEntry),
				Status: ResStatusProcessing,
				Type:   TypePerson,
				Value:  person.Login,
			}
			processedResolutions = append(processedResolutions, r.ResolveImp(res)...)
		}
	}

	return processedResolutions
}

func (r Resolver) ResolveBotInstanceNumber(resolution Resolution) []Resolution {
	var processedResolutions []Resolution

	supplierID, err := r.ABC.GetSupplierIDBySlug("bot")

	if err != nil {
		flowEntry := FlowEntry{
			Resolution: fmt.Sprintf("Resolver::ResolveBotInstanceNumber. Error in resolving abc supplier: %v", err),
			Resolver:   ResolverABCResource,
			Value:      resolution.Value,
		}
		res := Resolution{
			Flow:   append(resolution.Flow, flowEntry),
			Status: ResStatusFail,
			Type:   TypeBotInstanceNumber,
			Value:  resolution.Value,
		}
		processedResolutions = append(processedResolutions, res)
		return processedResolutions
	}

	consumerSlugs, err := r.ABC.GetConsumerSlugsByResourceExternalIDAndSupplierID(resolution.Value, *supplierID)

	if err != nil {
		flowEntry := FlowEntry{
			Resolution: fmt.Sprintf("error: %v", err),
			Resolver:   ResolverABCResource,
			Value:      resolution.Value,
		}
		res := Resolution{
			Flow:   append(resolution.Flow, flowEntry),
			Status: ResStatusFail,
			Type:   TypeBotInstanceNumber,
			Value:  resolution.Value,
		}
		processedResolutions = append(processedResolutions, res)
		return processedResolutions
	}

	flowEntry := FlowEntry{
		Resolution: fmt.Sprintf("Consumer of abc resource with external id (%s)", resolution.Value),
		Resolver:   ResolverABCResource,
		Value:      resolution.Value,
	}
	flow := append(resolution.Flow, flowEntry)
	for _, slug := range consumerSlugs {
		res := Resolution{
			Flow:   flow,
			Status: ResStatusProcessing,
			Type:   TypeService,
			Value:  slug,
		}
		processedResolutions = append(processedResolutions, r.ResolveImp(res)...)
	}
	return processedResolutions
}

func (r Resolver) ResolveTVM(resolution Resolution) []Resolution {
	var processedResolutions []Resolution

	resourceTypeID, err := r.ABC.GetResourceTypeID("tvm_app")

	if err != nil {
		flowEntry := FlowEntry{
			Resolution: fmt.Sprintf("Resolver::ResolveTVM. Error in resolving abc resource type: %v", err),
			Resolver:   ResolverTVM,
			Value:      resolution.Value,
		}
		res := Resolution{
			Flow:   append(resolution.Flow, flowEntry),
			Status: ResStatusFail,
			Type:   TypeTVM,
			Value:  resolution.Value,
		}
		processedResolutions = append(processedResolutions, res)
		return processedResolutions
	}

	consumerSlugs, err := r.ABC.GetConsumerSlugsByResourceExternalIDAndResourceType(resolution.Value, *resourceTypeID)

	if err != nil {
		flowEntry := FlowEntry{
			Resolution: fmt.Sprintf("error: %v", err),
			Resolver:   ResolverTVM,
			Value:      resolution.Value,
		}
		res := Resolution{
			Flow:   append(resolution.Flow, flowEntry),
			Status: ResStatusFail,
			Type:   TypeTVM,
			Value:  resolution.Value,
		}
		processedResolutions = append(processedResolutions, res)
		return processedResolutions
	}

	flowEntry := FlowEntry{
		Resolution: fmt.Sprintf("Consumer of abc resource with external id (%s)", resolution.Value),
		Resolver:   ResolverTVM,
		Value:      resolution.Value,
	}
	flow := append(resolution.Flow, flowEntry)
	for _, slug := range consumerSlugs {
		res := Resolution{
			Flow:   flow,
			Status: ResStatusProcessing,
			Type:   TypeService,
			Value:  slug,
		}
		processedResolutions = append(processedResolutions, r.ResolveImp(res)...)
	}
	return processedResolutions
}

func (r Resolver) ResolveImp(resolution Resolution) []Resolution {

	if resolution.Status != ResStatusProcessing {
		return []Resolution{resolution}
	} else if resolution.GetDepth() >= r.ResolverMaxDepth {
		return []Resolution{MaxDepthExceededResolution(resolution)}
	} else {
		for _, flow := range resolution.Flow {
			if flow.Value == resolution.Value {
				return []Resolution{RecursionBlockResolution(resolution)}
			}
		}
	}

	var processedResolutions []Resolution

	sentToResolutionCounter := 0

	resolutionsChannel := make(chan []Resolution)

	if resolution.IsPerson() {
		sentToResolutionCounter++
		go func() { resolutionsChannel <- r.ResolvePerson(resolution) }()
	}

	if resolution.IsServiceRole() {
		sentToResolutionCounter++
		go func() { resolutionsChannel <- r.ResolveServiceRole(resolution) }()
	}

	if resolution.IsDepartment() {
		sentToResolutionCounter++
		go func() { resolutionsChannel <- r.ResolveDepartment(resolution) }()
	}

	if resolution.IsWikiGroup() {
		sentToResolutionCounter++
		go func() { resolutionsChannel <- r.ResolveWikiGroup(resolution) }()
	}

	if resolution.IsService() {
		sentToResolutionCounter++
		go func() { resolutionsChannel <- r.ResolveService(resolution) }()
	}

	if resolution.IsIPAddress() {
		sentToResolutionCounter++
		go func() { resolutionsChannel <- r.ResolveIPAddressToMACAddress(resolution) }()
		sentToResolutionCounter++
		go func() { resolutionsChannel <- r.ResolveIPAddressToMacro(resolution) }()
		sentToResolutionCounter++
		go func() { resolutionsChannel <- r.ResolveIPToFQDN(resolution) }()
		sentToResolutionCounter++
		go func() { resolutionsChannel <- r.ResolveIPAddressToNetwork(resolution) }()
		// sentToResolutionCounter++
		// go func() { resolutionsChannel <- r.ResolveIPToPuncherResps(resolution) }()
	}

	if resolution.IsMACAddress() {
		sentToResolutionCounter++
		go func() { resolutionsChannel <- r.ResolveMACAddress(resolution) }()
	}

	if resolution.IsBotInstanceNumber() {
		sentToResolutionCounter++
		go func() { resolutionsChannel <- r.ResolveBotInstanceNumber(resolution) }()
	}

	if resolution.IsTVM() {
		sentToResolutionCounter++
		go func() { resolutionsChannel <- r.ResolveTVM(resolution) }()
	}

	if resolution.IsFQDN() {
		sentToResolutionCounter++
		go func() { resolutionsChannel <- r.ResolveFQDNRacktablesOwners(resolution) }()
		sentToResolutionCounter++
		go func() { resolutionsChannel <- r.ResolveFQDNWalle(resolution) }()
		sentToResolutionCounter++
		go func() { resolutionsChannel <- r.ResolveFQDNToIP(resolution) }()
		// sentToResolutionCounter++
		// go func() { resolutionsChannel <- r.ResolveFQDNToPuncherResps(resolution) }()
	}

	if resolution.IsMacro() {
		sentToResolutionCounter++
		go func() { resolutionsChannel <- r.ResolveMacro(resolution) }()
		// sentToResolutionCounter++
		// go func() { resolutionsChannel <- r.ResolveMacroToPuncherResps(resolution) }()
	}

	if resolution.IsNetwork() {
		sentToResolutionCounter++
		go func() { resolutionsChannel <- r.ResolveNetwork(resolution) }()
	}

	if resolution.IsServiceID() {
		sentToResolutionCounter++
		go func() { resolutionsChannel <- r.ResolveServiceID(resolution) }()
	}

	if sentToResolutionCounter == 0 {
		return []Resolution{UnsupportedResolution(resolution)}
	}

	for i := 0; i < sentToResolutionCounter; i++ {
		resolutions := <-resolutionsChannel
		processedResolutions = append(processedResolutions, resolutions...)
	}

	return processedResolutions
}

func (r Resolver) Resolve(value string) []Resolution {
	query := Resolution{
		Flow:   nil,
		Status: ResStatusProcessing,
		Type:   TypeUnknown,
		Value:  value,
	}

	return r.ResolveImp(query)
}

func (r Resolver) ResolveType(inputValue string, inputType ValueType) []Resolution {
	query := Resolution{
		Flow:   nil,
		Status: ResStatusProcessing,
		Type:   inputType,
		Value:  inputValue,
	}

	return r.ResolveImp(query)
}
