package staff

import (
	"fmt"
)

type (
	ResStatus string
)

type StaffResolution struct {
	ResolutionStatus ResStatus `json:"resolution_status"`
	ResolveDepth     int       `json:"resolve_depth"`
	ResolveFlow      string    `json:"resolve_flow"`
	User             string    `json:"user"`
}

const (
	ResolutionFail       ResStatus = "ResolutionFail"
	ResolutionProcessing ResStatus = "ResolutionProcessing"
	ResolutionSuccess    ResStatus = "ResolutionSuccess"
)

func IsAllResolutionsProcessed(resolutions []StaffResolution) bool {
	for _, resolution := range resolutions {
		if resolution.ResolutionStatus == ResolutionProcessing {
			return false
		}
	}
	return true
}

func (staff Staff) ResolveImpl(resolutions []StaffResolution) []StaffResolution {
	var processedResolutions []StaffResolution

	if IsAllResolutionsProcessed(resolutions) {
		return resolutions
	}

	for _, resolution := range resolutions {

		if resolution.ResolutionStatus != ResolutionProcessing {
			processedResolutions = append(processedResolutions, resolution)

		} else if resolution.ResolveDepth == staff.ResolverMaxDepth {
			res := StaffResolution{
				ResolutionStatus: ResolutionFail,
				ResolveDepth:     resolution.ResolveDepth,
				ResolveFlow:      fmt.Sprintf("Reached max depth for (%s)", resolution.ResolveFlow),
				User:             resolution.User,
			}
			processedResolutions = append(processedResolutions, res)

		} else {

			staffPerson, err := staff.GetUserInfo(resolution.User)

			if err != nil {
				res := StaffResolution{
					ResolutionStatus: ResolutionFail,
					ResolveDepth:     resolution.ResolveDepth + 1,
					ResolveFlow:      fmt.Sprintf("Did not find (%s)", resolution.ResolveFlow),
					User:             resolution.User,
				}
				processedResolutions = append(processedResolutions, res)

			} else if !staffPerson.IsFired() && !staffPerson.IsRobot() {
				res := StaffResolution{
					ResolutionStatus: ResolutionSuccess,
					ResolveDepth:     resolution.ResolveDepth + 1,
					ResolveFlow:      fmt.Sprintf("Resolved (%s)", resolution.ResolveFlow),
					User:             resolution.User,
				}
				processedResolutions = append(processedResolutions, res)

			} else if staffPerson.IsRobot() {
				for _, robotOwner := range staffPerson.RobotOwners {
					res := StaffResolution{
						ResolutionStatus: ResolutionProcessing,
						ResolveDepth:     resolution.ResolveDepth + 1,
						ResolveFlow:      fmt.Sprintf("Owner of robot (%s)", resolution.ResolveFlow),
						User:             robotOwner.Person.Login,
					}
					processedResolutions = append(processedResolutions, res)
				}

			} else if staffPerson.IsFired() {
				for _, departmentHead := range staffPerson.DepartmentGroup.Department.Heads {
					res := StaffResolution{
						ResolutionStatus: ResolutionProcessing,
						ResolveDepth:     resolution.ResolveDepth + 1,
						ResolveFlow:      fmt.Sprintf("Head with role '%s' of department of fired (%s)", departmentHead.Role, resolution.ResolveFlow),
						User:             departmentHead.Person.Login,
					}
					processedResolutions = append(processedResolutions, res)
				}

			} else {
				res := StaffResolution{
					ResolutionStatus: ResolutionFail,
					ResolveDepth:     resolution.ResolveDepth + 1,
					ResolveFlow:      fmt.Sprintf("Unknown problem in resolving (%s)", resolution.ResolveFlow),
					User:             resolution.User,
				}
				processedResolutions = append(processedResolutions, res)
			}
		}
	}

	return staff.ResolveImpl(processedResolutions)
}

func (staff Staff) Resolve(user string) []StaffResolution {
	resolution := StaffResolution{
		ResolutionStatus: ResolutionProcessing,
		ResolveDepth:     0,
		ResolveFlow:      user,
		User:             user,
	}
	resolutions := []StaffResolution{resolution}
	return staff.ResolveImpl(resolutions)
}
