package staff

import (
	"crypto/tls"
	"fmt"
	"strings"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/certifi"
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/log/nop"
)

type (
	GroupType string
)

const (
	GroupTypeDepartment  GroupType = "department"
	GroupTypeWiki        GroupType = "wiki"
	GroupTypeServicerole GroupType = "servicerole"
)

var (
	personFields = []string{
		"id", "login", "robot_owners", "official.is_dismissed", "official.is_robot",
		"department_group.id", "department_group.name", "department_group.parent.id",
		"department_group.parent.name", "department_group.parent.url", "department_group.url",
		"department_group.department.heads",
	}

	DefaultResolverMaxDepth = 10
)

type Staff struct {
	Token            string
	ResolverMaxDepth int
	Client           resty.Client
	Logger           log.Logger
}

func NewHTTPClient(token string) (*resty.Client, error) {
	certPool, err := certifi.NewCertPool()
	if err != nil {
		return nil, err
	}
	httpClient := resty.New().
		SetRetryCount(2).
		SetHeader("Content-Type", "application/json").
		SetHeader("Authorization", "OAuth "+token).
		SetHeader("User-Agent", "ya_resolve <security@yandex-team.ru>").
		SetTLSClientConfig(&tls.Config{RootCAs: certPool})
	return httpClient, nil
}

func NewStaff(token string, options ...Option) (*Staff, error) {
	httpClient, err := NewHTTPClient(token)
	if err != nil {
		return nil, err
	}

	staff := Staff{
		Token:            token,
		ResolverMaxDepth: DefaultResolverMaxDepth,
		Client:           *httpClient,
		Logger:           &nop.Logger{},
	}

	for _, opt := range options {
		opt(&staff)
	}

	return &staff, nil
}

func (staffPerson StaffPerson) IsRobot() bool {
	return staffPerson.Official.IsRobot
}

func (staffPerson StaffPerson) IsFired() bool {
	return staffPerson.Official.IsDismissed
}

func (staff Staff) GetGroupMembership(value string, groupType GroupType) ([]StaffPerson, error) {
	var staffResponse StaffResponse

	resp, err := staff.Client.R().
		SetQueryParams(map[string]string{
			"group.url":  value,
			"group.type": string(groupType),
			"_fields":    "person",
		}).
		SetResult(&staffResponse).
		Get("https://staff-api.yandex-team.ru/v3/groupmembership")

	if err != nil {
		return nil, err
	}

	if !resp.IsSuccess() {
		return nil, fmt.Errorf("Staff::GetWikiGroupMembers. resp.StatusCode != 200. Status code: %d", resp.StatusCode())
	}

	var persons = make([]StaffPerson, len(staffResponse.Result))
	for i, result := range staffResponse.Result {
		persons[i] = result.Person
	}

	return persons, nil
}

func (staff Staff) GetServiceRoleMembers(servicerole string) ([]StaffPerson, error) {
	return staff.GetGroupMembership(servicerole, GroupTypeServicerole)
}

func (staff Staff) GetDepartmentMembers(department string) ([]StaffPerson, error) {
	return staff.GetGroupMembership(department, GroupTypeDepartment)
}

func (staff Staff) GetWikiGroupMembers(wikiGroup string) ([]StaffPerson, error) {
	return staff.GetGroupMembership(wikiGroup, GroupTypeWiki)
}

func (staff Staff) GetUserInfo(user string) (*StaffPerson, error) {

	var staffPerson StaffPerson

	resp, err := staff.Client.R().
		SetQueryParams(map[string]string{
			"login":   user,
			"_one":    "1",
			"_fields": strings.Join(personFields, ","),
		}).
		SetResult(&staffPerson).
		Get("https://staff-api.yandex-team.ru/v3/persons")

	if err != nil {
		return nil, err
	}

	if !resp.IsSuccess() {
		return nil, fmt.Errorf("Staff::GetUserInfo. resp.StatusCode != 200. Status code: %d", resp.StatusCode())
	}

	return &staffPerson, nil
}
