package walle

import (
	"fmt"
)

const (
	HostsLimit = 10000
)

type ProjectInfo struct {
	BotProjectID int      `json:"bot_project_id"`
	ID           string   `json:"id"`
	Name         string   `json:"name"`
	Owners       []string `json:"owners"`
	Provisioner  string   `json:"provisioner"`
}

type HostInfo struct {
	ActiveMAC    string   `json:"active_mac"`
	Inv          int      `json:"inv"`
	IPMIMAC      string   `json:"ipmi_mac"`
	MACs         []string `json:"mac"`
	Name         string   `json:"name"`
	Project      string   `json:"project"`
	StateAuthor  string   `json:"state_author"`
	StatusAuthor string   `json:"status_author"`
	UUID         string   `json:"uuid"`
}

type ProjectsResponse struct {
	Result []ProjectInfo `json:"result"`
}

type HostsResponse struct {
	NextCursor int64      `json:"next_cursor"`
	Result     []HostInfo `json:"result"`
}

func (c Client) GetProjects() ([]ProjectInfo, error) {
	var projectsResponse ProjectsResponse

	resp, err := c.Client.R().
		SetQueryParams(map[string]string{
			"fields": "bot_project_id,id,name,owners,provisioner",
		}).
		SetResult(&projectsResponse).
		Get("https://api.wall-e.yandex-team.ru/v1/projects")

	if err != nil {
		return nil, err
	}

	if !resp.IsSuccess() {
		return nil, fmt.Errorf("Walle::GetProjects. Response status code != 200: %d", resp.StatusCode())
	}

	return projectsResponse.Result, nil
}

func (c Client) GetHostsWithCursor(cursor int64) (*HostsResponse, error) {
	var hostsResponse HostsResponse

	resp, err := c.Client.R().
		SetQueryParams(map[string]string{
			"fields": "active_mac,inv,ipmi_mac,macs,name,project,state_author,status_author,uuid",
			"limit":  fmt.Sprintf("%d", HostsLimit),
			"cursor": fmt.Sprintf("%d", cursor),
		}).
		SetResult(&hostsResponse).
		Get("https://api.wall-e.yandex-team.ru/v1/hosts")

	if err != nil {
		return nil, err
	}

	if !resp.IsSuccess() {
		return nil, fmt.Errorf("Walle::GetHostsWithCursor. Response status code != 200: %d", resp.StatusCode())
	}

	return &hostsResponse, nil

}

func (c Client) GetHosts() ([]HostInfo, error) {
	var allHosts []HostInfo
	var cursor int64

	cursor = 0

	for {
		resp, err := c.GetHostsWithCursor(cursor)
		if err != nil {
			c.Logger.Errorf("Walle:GetHosts. %v", err)
			return allHosts, err
		}

		cursor = resp.NextCursor
		hosts := resp.Result
		allHosts = append(allHosts, hosts...)

		if len(hosts) < HostsLimit {
			break
		}

		if cursor == 0 {
			break
		}
	}

	return allHosts, nil
}
