package cache

import (
	"context"
	"fmt"
	"time"

	"a.yandex-team.ru/security/ya_resolve/internal/walle"
)

type Host struct {
	Name               string
	StateAuthor        string
	StatusAuthor       string
	ProjectID          string
	ProjectOwners      []string
	ProjectProvisioner string
}

type WalleCache struct {
	Client                 walle.Client
	HostsDict              map[string]Host
	PeriodicSyncingContext context.Context
	ContextCancelFunc      context.CancelFunc
	LastSyncTimestamp      int64
	PeriodSyncingIsOn      bool
}

func NewCache(client walle.Client) WalleCache {
	return WalleCache{
		Client: client,
	}
}

func GetProjectByID(projects []walle.ProjectInfo, projectID string) (*walle.ProjectInfo, error) {
	for _, project := range projects {
		if project.ID == projectID {
			return &project, nil
		}
	}
	return nil, fmt.Errorf("WalleCache:GetProjectByID. Project with id %s not found", projectID)
}

func (c WalleCache) GetHostInfoByFQDN(fqdn string) (*Host, error) {
	host, ok := c.HostsDict[fqdn]
	if !ok {
		return nil, fmt.Errorf("WalleCache:GetHostInfoByFQDN. fqdn not found")
	} else {
		return &host, nil
	}
}

func (c *WalleCache) Sync() {
	c.updateLastSyncTimestamp()

	c.Client.Logger.Infof("WalleCache:Sync. Started")

	projects, err := c.Client.GetProjects()
	if err != nil {
		c.Client.Logger.Warnf("WalleCache:Sync. Error syncing projects: %v.\n", err)
		return
	}

	hosts, err := c.Client.GetHosts()
	if err != nil {
		c.Client.Logger.Warnf("WalleCache:Sync. Error syncing hosts: %v.\n", err)
		return
	}

	hostsDict := make(map[string]Host)

	for _, host := range hosts {
		project, err := GetProjectByID(projects, host.Project)
		if err != nil {
			c.Client.Logger.Warnf("WalleCache:Sync. host: %#v. error: %v", host, err)
			hostsDict[host.Name] = Host{
				Name:         host.Name,
				StateAuthor:  host.StateAuthor,
				StatusAuthor: host.StatusAuthor,
			}
		} else {
			hostsDict[host.Name] = Host{
				Name:               host.Name,
				StateAuthor:        host.StateAuthor,
				StatusAuthor:       host.StatusAuthor,
				ProjectID:          project.ID,
				ProjectOwners:      project.Owners,
				ProjectProvisioner: project.Provisioner,
			}
		}
	}

	c.HostsDict = hostsDict
	c.Client.Logger.Infof("WalleCache:Sync. Successfully finished")
}

func (c *WalleCache) updateLastSyncTimestamp() {
	c.LastSyncTimestamp = time.Now().Unix()
}

func (c *WalleCache) StartPeriodicSyncing(periodSeconds int64) bool {
	if c.PeriodSyncingIsOn {
		return false
	}

	c.PeriodSyncingIsOn = true

	syncer := func(ctx context.Context) {
		for {
			select {
			case <-ctx.Done():
				return
			default:
				if time.Now().Unix()-c.LastSyncTimestamp > periodSeconds {
					c.Sync()
				} else {
					time.Sleep(1 * time.Second)
				}
			}
		}
	}
	c.PeriodicSyncingContext, c.ContextCancelFunc = context.WithCancel(context.Background())

	go syncer(c.PeriodicSyncingContext)

	return true
}

func (c *WalleCache) StopPeriodicSyncing() bool {
	if c.PeriodSyncingIsOn {
		c.ContextCancelFunc()
		c.PeriodSyncingIsOn = false
		return true
	} else {
		return false
	}
}
